<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\GeocodingController;

// Registrar el componente Livewire
Livewire::component('admin.empleados.order-notifications', \App\Livewire\Admin\Empleados\OrderNotifications::class);

// Rutas de geocodificación
Route::prefix('api/geocoding')->group(function () {
    Route::get('/reverse', [GeocodingController::class, 'reverse'])->name('geocoding.reverse');
    Route::get('/search', [GeocodingController::class, 'search'])->name('geocoding.search');
    Route::get('/nominatim', [GeocodingController::class, 'nominatim'])->name('geocoding.nominatim');
    Route::get('/opencage', [GeocodingController::class, 'openCage'])->name('geocoding.opencage');
    Route::post('/save-accuracy', [GeocodingController::class, 'saveLocationAccuracy'])->name('geocoding.save-accuracy');
    Route::get('/get-accuracy', [GeocodingController::class, 'getLocationAccuracy'])->name('geocoding.get-accuracy');
});

Route::get('/test-send-message', function () {
    try {
        $response = Http::withHeaders([
            'X-API-Key' => 'test-api-key-vargas-centro',
            'Content-Type' => 'application/json'
        ])->timeout(30)->post('http://localhost:3001/api/whatsapp/send', [
            'to' => '584121234567',
            'message' => 'Mensaje de prueba desde Laravel',
            'type' => 'text'
        ]);
        
        return response()->json([
            'success' => $response->successful(),
            'status' => $response->status(),
            'body' => $response->json()
        ]);
    } catch (\Exception $e) {
        return response()->json(['error' => $e->getMessage()]);
    }
});


// Rutas de la front (existentes)
Route::prefix('front')->name('front.')->group(function () {
    // Ruta principal del store
    Route::get('/store', \App\Livewire\Front\Index::class)->name('store');
    Route::get('/cart', \App\Livewire\Front\Cart\Index::class)->name('cart');
    Route::get('/checkout', \App\Livewire\Front\Checkout\Index::class)->name('checkout');
    Route::get('/wishlist', \App\Livewire\Front\Wishlist\Index::class)->name('wishlist');
    Route::get('/pedido/historico', \App\Livewire\Front\Pedido\Historico::class)->name('pedido.historico');
    Route::get('/pedido/detalle/{codigo}', \App\Livewire\Front\Pedido\Detalle::class)->name('pedido.detalle');
    Route::get('/pedido/resena/{id}', \App\Livewire\Front\Resena\Index::class)->name('pedido.resena');
    
    // Perfil del usuario
    Route::get('/perfil', \App\Livewire\Front\User\Profile::class)->name('perfil')->middleware('auth');
    
    // Ruta de prueba para QuickView
    Route::get('/quickview-test', function () {
        return view('livewire.front.quick-view-test');
    })->name('quickview-test');
});

Route::get('/lang/{locale}', function ($locale) {
    if (in_array($locale, ['es', 'en'])) {
        Session::put('locale', $locale);
    }
    return redirect()->back();
})->name('lang.switch');

Route::get('/', function () {
   if (\Auth::check() && \Auth::user()->hasRole('SuperAdmin')) {

    return redirect()->to('superadmin/dashboard');
   } elseif (\Auth::check() && \Auth::user()->hasRole('Administrador')) {
    # code...
    return redirect()->to('admin/dashboard');
   } else {
    if (\Auth::check() && \Auth::user()->hasRole('Empleado')) {
         # code...
    return redirect()->to('admin/empleados/dashboard');
    }
    else
    {
       return redirect()->to('front/store');
    }
   }
});

// Rutas de autenticación con Livewire
Route::middleware('guest')->group(function () {
    Route::get('login', \App\Livewire\Auth\Login::class)->name('login');
    Route::get('register', \App\Livewire\Auth\Register::class)->name('register');
    Route::get('password/reset', \App\Livewire\Auth\ForgotPassword::class)->name('password.request');
    Route::get('password/reset/{token}', \App\Livewire\Auth\ResetPassword::class)->name('password.reset');
});

Route::middleware('auth')->group(function () {
    Route::get('verify-email', \App\Livewire\Auth\VerifyCode::class)->name('verification.notice');
    Route::get('verify-email/{id}/{hash}', function (string $id, string $hash) {
        // Esta ruta se usa para la verificación real del correo electrónico
        // Pero como estamos usando Livewire, simplemente redirigimos al componente Verify
        return redirect()->route('verification.notice');
    })->middleware(['signed', 'throttle:6,1'])->name('verification.verify');
});

// Ruta de logout
Route::post('logout', function () {
   if (auth()->user()->hasRole('Super Administrador') || auth()->user()->hasRole('Administrador') || auth()->user()->hasRole('Empleado') || auth()->user()->hasRole('Recepcionista')) {
        auth()->logout();
        request()->session()->invalidate();
        request()->session()->regenerateToken();

    return redirect('/login');;
   } else {
        auth()->logout();
        request()->session()->invalidate();
        request()->session()->regenerateToken();

        return redirect('/');
   }

})->name('logout');

// Ruta para verificación 2FA
Route::get('/two-factor-login', \App\Livewire\Auth\TwoFactorLogin::class)->name('two-factor.login');

// Super Admin routes
Route::group(['prefix' => 'superadmin', 'as' => 'superadmin.', 'middleware' => ['auth', 'verified']], function () {
    Route::get('/dashboard', \App\Livewire\SuperAdmin\Dashboard::class)->name('dashboard');
});

// Admin routes
Route::group(['prefix' => 'admin', 'as' => 'admin.', 'middleware' => ['auth', 'verified']], function () {
    Route::get('/dashboard', \App\Livewire\Admin\Dashboard::class)->name('dashboard');

    require __DIR__.'/admin.php';
});