<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>WhatsApp Chat - React</title>
    
    <link rel="stylesheet" href="https://fonts.googleapis.com/css?family=Roboto:300,400,500,700&display=swap" />
    <link rel="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons" />
    
    <script crossorigin src="https://unpkg.com/react@18/umd/react.development.js"></script>
    <script crossorigin src="https://unpkg.com/react-dom@18/umd/react-dom.development.js"></script>
    <script src="https://unpkg.com/@babel/standalone/babel.min.js"></script>
    <script src="https://unpkg.com/@material-ui/core@4.12.4/umd/material-ui.development.js"></script>
    <script src="https://unpkg.com/axios/dist/axios.min.js"></script>
    
    <style>
        body { margin: 0; font-family: 'Roboto', sans-serif; }
        #whatsapp-chat { height: 100vh; }
        .status-card { width: 300px; padding: 20px; text-align: center; }
        .chat-card { flex: 1; padding: 20px; }
        .messages-box { height: 300px; overflow-y: auto; margin-bottom: 16px; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
        .message { display: flex; margin-bottom: 8px; }
        .message.sent { justify-content: flex-end; }
        .message-bubble { padding: 8px; max-width: 70%; border-radius: 8px; }
        .message-bubble.sent { background-color: #25d366; color: white; }
        .message-bubble.received { background-color: #f5f5f5; color: black; }
        .input-row { display: flex; gap: 8px; }
        .error { padding: 16px; background-color: #ffebee; color: #c62828; margin: 8px; }
    </style>
</head>
<body>
    <div id="whatsapp-chat"></div>

    <script type="text/babel">
        const { useState, useEffect } = React;
        const { 
            ThemeProvider, 
            createMuiTheme, 
            CssBaseline, 
            Box, 
            Paper, 
            Typography, 
            Button, 
            TextField, 
            AppBar,
            Toolbar,
            Card,
            CardContent
        } = MaterialUI;

        const theme = createMuiTheme({
            palette: {
                primary: { main: '#25d366' },
                secondary: { main: '#128c7e' },
            },
        });

        function WhatsAppChat() {
            const [status, setStatus] = useState('disconnected');
            const [qrCode, setQrCode] = useState(null);
            const [selectedContact, setSelectedContact] = useState('');
            const [message, setMessage] = useState('');
            const [messages, setMessages] = useState([]);
            const [loading, setLoading] = useState(false);
            const [error, setError] = useState('');

            const checkStatus = async () => {
                setLoading(true);
                try {
                    const response = await axios.get('http://localhost:3000/api/whatsapp/status');
                    if (response.data.success) {
                        setStatus(response.data.status);
                        setQrCode(response.data.qr);
                    }
                } catch (err) {
                    setError('Error conectando con WhatsApp API');
                } finally {
                    setLoading(false);
                }
            };

            const sendMessage = async () => {
                if (!message.trim() || !selectedContact) return;

                try {
                    const response = await axios.post('http://localhost:3000/api/whatsapp/send-message', {
                        phone: selectedContact,
                        message: message
                    });

                    if (response.data.success) {
                        setMessages(prev => [...prev, {
                            type: 'sent',
                            text: message,
                            timestamp: new Date()
                        }]);
                        setMessage('');
                    }
                } catch (err) {
                    setError('Error enviando mensaje');
                }
            };

            useEffect(() => {
                checkStatus();
                const interval = setInterval(checkStatus, 5000);
                return () => clearInterval(interval);
            }, []);

            return (
                <ThemeProvider theme={theme}>
                    <CssBaseline />
                    <div style={{ height: '100vh', display: 'flex', flexDirection: 'column' }}>
                        <AppBar position="static">
                            <Toolbar>
                                <i className="material-icons" style={{ marginRight: 16 }}>chat</iconify-icon>
                                <Typography variant="h6" style={{ flexGrow: 1 }}>
                                    WhatsApp Business
                                </Typography>
                                <Button color="inherit" onClick={checkStatus} disabled={loading}>
                                    <i className="material-icons" style={{ marginRight: 8 }}>refresh</iconify-icon>
                                    {loading ? 'Cargando...' : 'Actualizar'}
                                </Button>
                            </Toolbar>
                        </AppBar>

                        {error && (
                            <div className="error">
                                {error}
                                <Button onClick={() => setError('')} style={{ marginLeft: 16 }}>×</Button>
                            </div>
                        )}

                        <div style={{ flex: 1, display: 'flex', padding: 16, gap: 16 }}>
                            <Card className="status-card">
                                <CardContent>
                                    {status === 'ready' ? (
                                        <>
                                            <i className="material-icons" style={{ fontSize: 60, color: '#4caf50', marginBottom: 16 }}>check_circle</iconify-icon>
                                            <Typography variant="h6" style={{ color: '#4caf50' }}>
                                                Conectado
                                            </Typography>
                                            <Typography variant="body2" color="textSecondary">
                                                WhatsApp está listo
                                            </Typography>
                                        </>
                                    ) : status === 'qr' ? (
                                        <>
                                            <Typography variant="h6" style={{ color: '#ff9800', marginBottom: 16 }}>
                                                Escanea el QR
                                            </Typography>
                                            {qrCode && (
                                                <img 
                                                    src={`data:image/png;base64,${qrCode}`} 
                                                    alt="QR Code" 
                                                    style={{ maxWidth: '200px', width: '100%' }}
                                                />
                                            )}
                                        </>
                                    ) : (
                                        <>
                                            <i className="material-icons" style={{ fontSize: 60, color: '#9e9e9e', marginBottom: 16 }}>phone</iconify-icon>
                                            <Typography variant="h6" color="textSecondary">
                                                Desconectado
                                            </Typography>
                                        </>
                                    )}
                                </CardContent>
                            </Card>

                            {status === 'ready' && (
                                <Card className="chat-card">
                                    <CardContent>
                                        <Typography variant="h6" style={{ marginBottom: 16 }}>
                                            Chat WhatsApp
                                        </Typography>
                                        
                                        <TextField
                                            fullWidth
                                            label="Número de teléfono"
                                            value={selectedContact}
                                            onChange={(e) => setSelectedContact(e.target.value)}
                                            placeholder="584121234567"
                                            style={{ marginBottom: 16 }}
                                        />

                                        <div className="messages-box">
                                            {messages.length === 0 ? (
                                                <Typography color="textSecondary" style={{ textAlign: 'center', marginTop: 100 }}>
                                                    No hay mensajes
                                                </Typography>
                                            ) : (
                                                messages.map((msg, index) => (
                                                    <div key={index} className={`message ${msg.type}`}>
                                                        <div className={`message-bubble ${msg.type}`}>
                                                            <Typography variant="body2">
                                                                {msg.text}
                                                            </Typography>
                                                        </div>
                                                    </div>
                                                ))
                                            )}
                                        </div>

                                        <div className="input-row">
                                            <TextField
                                                fullWidth
                                                label="Mensaje"
                                                value={message}
                                                onChange={(e) => setMessage(e.target.value)}
                                                onKeyPress={(e) => e.key === 'Enter' && sendMessage()}
                                                multiline
                                                rows={2}
                                            />
                                            <Button 
                                                variant="contained" 
                                                color="primary"
                                                onClick={sendMessage}
                                                disabled={!message.trim() || !selectedContact}
                                                style={{ minWidth: 60 }}
                                            >
                                                <i className="material-icons">send</iconify-icon>
                                            </Button>
                                        </div>
                                    </CardContent>
                                </Card>
                            )}
                        </div>
                    </div>
                </ThemeProvider>
            );
        }

        ReactDOM.render(<WhatsAppChat />, document.getElementById('whatsapp-chat'));
    </script>
</body>
</html>