/**
 * Sistema de Notificaciones Global
 * Basado en Materialize Template con Notyf
 */

'use strict';

// Configuración de colores basada en el template
const config = {
    colors: {
        primary: '#7367f0',
        secondary: '#a8aaae',
        success: '#28c76f',
        info: '#00cfe8',
        warning: '#ff9f43',
        danger: '#ea5455',
        dark: '#4b4b4b',
        black: '#000',
        white: '#fff',
        cardColor: '#fff',
        bodyBg: '#f8f7fa',
        bodyColor: '#6f6b7d',
        headingColor: '#5d596c',
        textMuted: '#a5a3ae',
        borderColor: '#dbdade'
    }
};

// Clase CustomNotyf para permitir contenido HTML
class CustomNotyf {
    constructor(options = {}) {
        this.notifications = [];
        this.defaultOptions = {
            duration: 3000,
            ripple: true,
            dismissible: false,
            position: { x: 'right', y: 'top' }
        };
        this.options = { ...this.defaultOptions, ...options };
        this.types = {
            info: {
                background: config.colors.info,
                icon: 'ri-information-fill',
                className: 'notyf__info'
            },
            warning: {
                background: config.colors.warning,
                icon: 'ri-alert-fill',
                className: 'notyf__warning'
            },
            success: {
                background: config.colors.success,
                icon: 'ri-checkbox-circle-fill',
                className: 'notyf__success'
            },
            error: {
                background: config.colors.danger,
                icon: 'ri-close-circle-fill',
                className: 'notyf__error'
            }
        };
        this.init();
    }

    init() {
        this.createContainer();
        this.addStyles();
    }

    createContainer() {
        const position = this.options.position;
        const positionClass = `notyf-${position.y}-${position.x}`;

        this.container = document.createElement('div');
        this.container.className = `notyf ${positionClass}`;
        document.body.appendChild(this.container);
    }

    addStyles() {
        if (document.getElementById('notyf-styles')) return;

        const styles = `
            .notyf {
                position: fixed;
                z-index: 9999;
                pointer-events: none;
                display: flex;
                flex-direction: column;
                align-items: flex-end;
                gap: 10px;
                padding: 20px;
            }

            .notyf-top-right { top: 0; right: 0; }
            .notyf-top-left { top: 0; left: 0; align-items: flex-start; }
            .notyf-bottom-right { bottom: 0; right: 0; justify-content: flex-end; }
            .notyf-bottom-left { bottom: 0; left: 0; align-items: flex-start; justify-content: flex-end; }

            .notyf__toast {
                background: white;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                padding: 16px;
                min-width: 300px;
                max-width: 400px;
                display: flex;
                align-items: center;
                gap: 12px;
                pointer-events: all;
                transform: translateX(100%);
                opacity: 0;
                transition: all 0.3s ease;
            }

            .notyf__toast.show {
                transform: translateX(0);
                opacity: 1;
            }

            .notyf__toast.hide {
                transform: translateX(100%);
                opacity: 0;
            }

            .notyf__icon {
                width: 24px;
                height: 24px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
                font-size: 14px;
                flex-shrink: 0;
            }

            .notyf__message {
                flex: 1;
                color: #333;
                font-size: 14px;
                line-height: 1.4;
            }

            .notyf__dismiss {
                background: none;
                border: none;
                color: #999;
                cursor: pointer;
                font-size: 18px;
                padding: 0;
                width: 20px;
                height: 20px;
                display: flex;
                align-items: center;
                justify-content: center;
                flex-shrink: 0;
            }

            .notyf__dismiss:hover {
                color: #666;
            }

            .notyf__success .notyf__icon { background: ${config.colors.success}; }
            .notyf__error .notyf__icon { background: ${config.colors.danger}; }
            .notyf__warning .notyf__icon { background: ${config.colors.warning}; }
            .notyf__info .notyf__icon { background: ${config.colors.info}; }

            @keyframes ripple {
                0% { transform: scale(0); opacity: 1; }
                100% { transform: scale(4); opacity: 0; }
            }

            .notyf__ripple {
                position: absolute;
                border-radius: 50%;
                background: rgba(255,255,255,0.6);
                animation: ripple 0.6s ease-out;
            }
        `;

        const styleSheet = document.createElement('style');
        styleSheet.id = 'notyf-styles';
        styleSheet.textContent = styles;
        document.head.appendChild(styleSheet);
    }

    show(options) {
        const notification = this.createNotification(options);
        this.notifications.push(notification);
        this.container.appendChild(notification.element);

        // Trigger animation
        setTimeout(() => {
            notification.element.classList.add('show');
        }, 10);

        // Auto-dismiss
        if (options.duration !== 0) {
            setTimeout(() => {
                this.dismiss(notification);
            }, options.duration);
        }

        return notification;
    }

    createNotification(options) {
        const type = this.types[options.type] || this.types.info;
        const element = document.createElement('div');
        element.className = `notyf__toast ${type.className}`;

        const iconHtml = options.icon !== false ? `<div class="notyf__icon"><i class="${type.icon}"></i></div>` : '';
        const dismissHtml = options.dismissible ? '<button class="notyf__dismiss">&times;</button>' : '';

        element.innerHTML = `
            ${iconHtml}
            <div class="notyf__message">${options.message}</div>
            ${dismissHtml}
        `;

        const notification = {
            element: element,
            options: options
        };

        // Add dismiss functionality
        if (options.dismissible) {
            const dismissBtn = element.querySelector('.notyf__dismiss');
            if (dismissBtn) {
                dismissBtn.addEventListener('click', () => this.dismiss(notification));
            }
        }

        // Add ripple effect
        if (options.ripple) {
            element.addEventListener('click', (e) => this.createRipple(e, element));
        }

        return notification;
    }

    createRipple(e, element) {
        const ripple = document.createElement('span');
        ripple.className = 'notyf__ripple';

        const rect = element.getBoundingClientRect();
        const size = Math.max(rect.width, rect.height);
        const x = e.clientX - rect.left - size / 2;
        const y = e.clientY - rect.top - size / 2;

        ripple.style.width = ripple.style.height = size + 'px';
        ripple.style.left = x + 'px';
        ripple.style.top = y + 'px';

        element.appendChild(ripple);

        setTimeout(() => {
            ripple.remove();
        }, 600);
    }

    dismiss(notification) {
        notification.element.classList.add('hide');

        setTimeout(() => {
            if (notification.element.parentNode) {
                notification.element.parentNode.removeChild(notification.element);
            }
            const index = this.notifications.indexOf(notification);
            if (index > -1) {
                this.notifications.splice(index, 1);
            }
        }, 300);
    }

    dismissAll() {
        this.notifications.slice().forEach(notification => {
            this.dismiss(notification);
        });
    }

    // Métodos de conveniencia
    success(message, options = {}) {
        return this.show({ type: 'success', message, ...options });
    }

    error(message, options = {}) {
        return this.show({ type: 'error', message, ...options });
    }

    warning(message, options = {}) {
        return this.show({ type: 'warning', message, ...options });
    }

    info(message, options = {}) {
        return this.show({ type: 'info', message, ...options });
    }
}

// Inicializar el sistema de notificaciones global
const notyf = new CustomNotyf({
    duration: 3000,
    ripple: true,
    dismissible: false,
    position: { x: 'right', y: 'top' }
});

// Función global notify compatible con Livewire
window.notify = function(type, message, options = {}) {
    // Si el mensaje es un objeto, extraer las propiedades
    if (typeof type === 'object') {
        const config = type;
        type = config.type || 'info';
        message = config.message || config.text || '';
        options = config.options || {};
    }

    // Validar tipo
    const validTypes = ['success', 'error', 'warning', 'info'];
    if (!validTypes.includes(type)) {
        type = 'info';
    }

    // Mostrar notificación
    return notyf[type](message, options);
};

// Listener global para eventos de Livewire
if (typeof Livewire !== 'undefined') {
    Livewire.on('notify', (data) => {
        if (Array.isArray(data) && data.length > 0) {
            const notification = data[0];
            notify(notification.type, notification.message, notification.options);
        } else if (typeof data === 'object') {
            notify(data.type, data.message, data.options);
        }
    });
}

// Función para limpiar todas las notificaciones
window.clearNotifications = function() {
    notyf.dismissAll();
};

// Función para cambiar la posición de las notificaciones
window.setNotificationPosition = function(x, y) {
    notyf.options.position = { x, y };
    // Recrear el contenedor con la nueva posición
    if (notyf.container) {
        notyf.container.remove();
    }
    notyf.createContainer();
};

// Función para cambiar la duración de las notificaciones
window.setNotificationDuration = function(duration) {
    notyf.options.duration = duration;
};

// Función para cambiar el comportamiento de las notificaciones
window.setNotificationOptions = function(options) {
    Object.assign(notyf.options, options);
};

// Log para confirmar que el script se cargó correctamente
console.log('✅ Sistema de notificaciones global cargado exitosamente');
