<?php

namespace App\Services;

use App\Models\Coupon;
use App\Models\CouponRedemption;
use App\Models\ExchangeRate;
use App\Models\Pedido;
use App\Models\PedidoDiscount;
use App\Models\Promotion;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PromotionEngine
{
    public function quote(Pedido $pedido, ?string $couponCode, User $user): array
    {
        $pedido->loadMissing(['detalles.producto.categoria']);

        $subtotalUsd = $this->calculateSubtotal($pedido);
        $exchangeRate = $this->getExchangeRate();
        $subtotalBs = $subtotalUsd * $exchangeRate;

        $appliedDiscounts = [];
        $errors = [];
        $totalDiscountUsd = 0;

        $automaticPromotions = Promotion::where('empresa_id', $pedido->empresaId)
            ->active()
            ->automatic()
            ->orderBy('priority', 'desc')
            ->with(['conditions', 'productos', 'categorias'])
            ->get();

        foreach ($automaticPromotions as $promotion) {
            $conditionResult = $this->evaluateConditions($promotion, $pedido);

            if ($conditionResult['passes']) {
                $discount = $this->calculateDiscount($promotion, $conditionResult['eligible_items']);

                if ($discount > 0) {
                    if ($promotion->max_discount_amount !== null) {
                        $discount = min($discount, (float) $promotion->max_discount_amount);
                    }

                    if (!$promotion->stackable && count($appliedDiscounts) > 0) {
                        continue;
                    }

                    $totalDiscountUsd += $discount;
                    $appliedDiscounts[] = [
                        'type' => 'automatic',
                        'promotion_id' => $promotion->id,
                        'coupon_id' => null,
                        'name' => $promotion->name,
                        'discount_type' => $promotion->discount_type,
                        'discount_value' => $promotion->discount_value,
                        'discount_amount_usd' => $discount,
                        'discount_amount_bs' => $discount * $exchangeRate,
                    ];
                }
            }
        }

        if ($couponCode) {
            $couponValidation = $this->validateCoupon($couponCode, $pedido->empresaId, $user);

            if (!$couponValidation['valid']) {
                $errors[] = $couponValidation['error'];
            } else {
                $coupon = $couponValidation['coupon'];
                $promotion = $coupon->promotion;

                if ($promotion) {
                    $conditionResult = $this->evaluateConditions($promotion, $pedido);

                    if ($conditionResult['passes']) {
                        $discount = $this->calculateDiscount($promotion, $conditionResult['eligible_items']);

                        if ($discount > 0) {
                            if ($promotion->max_discount_amount !== null) {
                                $discount = min($discount, (float) $promotion->max_discount_amount);
                            }

                            $totalDiscountUsd += $discount;
                            $appliedDiscounts[] = [
                                'type' => 'coupon',
                                'promotion_id' => $promotion->id,
                                'coupon_id' => $coupon->id,
                                'name' => $promotion->name,
                                'coupon_code' => $coupon->code,
                                'discount_type' => $promotion->discount_type,
                                'discount_value' => $promotion->discount_value,
                                'discount_amount_usd' => $discount,
                                'discount_amount_bs' => $discount * $exchangeRate,
                            ];
                        }
                    } else {
                        $errors[] = 'El cupón no cumple con las condiciones requeridas.';
                    }
                }
            }
        }

        $totalDiscountUsd = min($totalDiscountUsd, $subtotalUsd);
        $totalDiscountBs = $totalDiscountUsd * $exchangeRate;

        return [
            'subtotal_usd' => round($subtotalUsd, 2),
            'subtotal_bs' => round($subtotalBs, 2),
            'discount_total_usd' => round($totalDiscountUsd, 2),
            'discount_total_bs' => round($totalDiscountBs, 2),
            'final_total_usd' => round($subtotalUsd - $totalDiscountUsd, 2),
            'final_total_bs' => round($subtotalBs - $totalDiscountBs, 2),
            'applied_discounts' => $appliedDiscounts,
            'errors' => $errors,
        ];
    }

    public function applyAndPersist(Pedido $pedido, ?string $couponCode): array
    {
        $user = $pedido->user;
        $quote = $this->quote($pedido, $couponCode, $user);

        if (!empty($quote['errors'])) {
            return [
                'success' => false,
                'errors' => $quote['errors'],
            ];
        }

        if (empty($quote['applied_discounts'])) {
            return [
                'success' => true,
                'message' => 'No hay descuentos aplicables.',
                'quote' => $quote,
            ];
        }

        try {
            DB::transaction(function () use ($pedido, $quote) {
                foreach ($quote['applied_discounts'] as $discount) {
                    PedidoDiscount::create([
                        'pedido_id' => $pedido->id,
                        'promotion_id' => $discount['promotion_id'],
                        'coupon_id' => $discount['coupon_id'] ?? null,
                        'discount_type' => $discount['discount_type'],
                        'discount_value' => $discount['discount_value'],
                        'discount_amount' => $discount['discount_amount_usd'],
                        'description' => $discount['name'],
                        'metadata' => [
                            'discount_amount_bs' => $discount['discount_amount_bs'],
                            'type' => $discount['type'],
                        ],
                    ]);

                    if (!empty($discount['coupon_id'])) {
                        CouponRedemption::create([
                            'coupon_id' => $discount['coupon_id'],
                            'pedido_id' => $pedido->id,
                            'user_id' => $pedido->userId,
                            'empresa_id' => $pedido->empresaId,
                            'discount_amount' => $discount['discount_amount_usd'],
                            'metadata' => [
                                'promotion_id' => $discount['promotion_id'],
                                'coupon_code' => $discount['coupon_code'] ?? null,
                            ],
                        ]);

                        Coupon::where('id', $discount['coupon_id'])
                            ->increment('usage_count');
                    }

                    Promotion::where('id', $discount['promotion_id'])
                        ->increment('usage_count');
                }

                $pedido->update([
                    'total_usd' => $quote['final_total_usd'],
                    'total_bs' => $quote['final_total_bs'],
                ]);
            });

            return [
                'success' => true,
                'message' => 'Descuentos aplicados correctamente.',
                'quote' => $quote,
            ];
        } catch (\Exception $e) {
            Log::error('Error applying discounts', [
                'pedido_id' => $pedido->id,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'errors' => ['Error al aplicar los descuentos: ' . $e->getMessage()],
            ];
        }
    }

    public function validateCoupon(string $code, $empresaId, User $user): array
    {
        $coupon = Coupon::where('code', $code)
            ->where('empresa_id', $empresaId)
            ->with('promotion')
            ->first();

        if (!$coupon) {
            return [
                'valid' => false,
                'error' => 'El cupón no existe.',
                'coupon' => null,
            ];
        }

        if (!$coupon->is_active) {
            return [
                'valid' => false,
                'error' => 'El cupón está inactivo.',
                'coupon' => null,
            ];
        }

        $now = Carbon::now();
        $startsAt = $coupon->getEffectiveStartsAt();
        $endsAt = $coupon->getEffectiveEndsAt();

        if ($startsAt && $now->lt($startsAt)) {
            return [
                'valid' => false,
                'error' => 'El cupón aún no está vigente.',
                'coupon' => null,
            ];
        }

        if ($endsAt && $now->gt($endsAt)) {
            return [
                'valid' => false,
                'error' => 'El cupón ha expirado.',
                'coupon' => null,
            ];
        }

        if ($coupon->usage_limit !== null && $coupon->usage_count >= $coupon->usage_limit) {
            return [
                'valid' => false,
                'error' => 'El cupón ha alcanzado su límite de usos.',
                'coupon' => null,
            ];
        }

        if ($coupon->usage_limit_per_user !== null) {
            $userRedemptions = CouponRedemption::where('coupon_id', $coupon->id)
                ->where('user_id', $user->id)
                ->count();

            if ($userRedemptions >= $coupon->usage_limit_per_user) {
                return [
                    'valid' => false,
                    'error' => 'Has alcanzado el límite de usos de este cupón.',
                    'coupon' => null,
                ];
            }
        }

        $promotion = $coupon->promotion;
        if ($promotion && !$promotion->isValid()) {
            return [
                'valid' => false,
                'error' => 'La promoción asociada al cupón no está disponible.',
                'coupon' => null,
            ];
        }

        return [
            'valid' => true,
            'error' => null,
            'coupon' => $coupon,
        ];
    }

    public function evaluateConditions(Promotion $promotion, Pedido $pedido): array
    {
        $promotion->loadMissing(['conditions', 'productos', 'categorias']);
        $pedido->loadMissing(['detalles.producto.categoria']);

        $allItems = $pedido->detalles;
        $eligibleItems = collect();
        $passes = true;

        $appliesTo = $promotion->applies_to ?? 'all';
        $promotionProductIds = $promotion->productos->pluck('id')->toArray();
        $promotionCategoryIds = $promotion->categorias->pluck('id')->toArray();

        foreach ($allItems as $detalle) {
            $isEligible = false;

            if ($appliesTo === 'all') {
                $isEligible = true;
            } elseif ($appliesTo === 'products' && in_array($detalle->productoId, $promotionProductIds)) {
                $isEligible = true;
            } elseif ($appliesTo === 'categories' && $detalle->producto?->categoria_id) {
                $isEligible = in_array($detalle->producto->categoria_id, $promotionCategoryIds);
            }

            if ($isEligible) {
                $eligibleItems->push($detalle);
            }
        }

        foreach ($promotion->conditions as $condition) {
            switch ($condition->type) {
                case 'min_amount':
                    $eligibleTotal = $eligibleItems->sum('total_usd');
                    if ($eligibleTotal < (float) $condition->value) {
                        $passes = false;
                    }
                    break;

                case 'products_in':
                    $requiredProductIds = $condition->metadata['product_ids'] ?? [];
                    if (!empty($requiredProductIds)) {
                        $cartProductIds = $allItems->pluck('productoId')->toArray();
                        $hasRequired = !empty(array_intersect($requiredProductIds, $cartProductIds));
                        if (!$hasRequired) {
                            $passes = false;
                        }
                    }
                    break;

                case 'categories_in':
                    $requiredCategoryIds = $condition->metadata['category_ids'] ?? [];
                    if (!empty($requiredCategoryIds)) {
                        $cartCategoryIds = $allItems
                            ->map(fn($d) => $d->producto?->categoria_id)
                            ->filter()
                            ->unique()
                            ->toArray();
                        $hasRequired = !empty(array_intersect($requiredCategoryIds, $cartCategoryIds));
                        if (!$hasRequired) {
                            $passes = false;
                        }
                    }
                    break;
            }

            if (!$passes) {
                break;
            }
        }

        return [
            'passes' => $passes,
            'eligible_items' => $eligibleItems,
        ];
    }

    public function calculateDiscount(Promotion $promotion, Collection $eligibleItems): float
    {
        if ($eligibleItems->isEmpty()) {
            return 0;
        }

        $discountType = $promotion->discount_type;
        $discountValue = (float) $promotion->discount_value;
        $eligibleAmount = $eligibleItems->sum('total_usd');

        switch ($discountType) {
            case 'percent':
                return $eligibleAmount * ($discountValue / 100);

            case 'fixed':
                return min($discountValue, $eligibleAmount);

            case 'bxgy':
                return $this->calculateBxgyDiscount($promotion, $eligibleItems);

            default:
                return 0;
        }
    }

    protected function calculateBxgyDiscount(Promotion $promotion, Collection $eligibleItems): float
    {
        $metadata = $promotion->metadata ?? [];
        $buyQty = $metadata['buy_qty'] ?? 1;
        $getQty = $metadata['get_qty'] ?? 1;

        $totalDiscount = 0;

        foreach ($eligibleItems as $item) {
            $qty = $item->quantity;
            $unitPrice = $qty > 0 ? ($item->total_usd / $qty) : 0;

            $setsCount = floor($qty / ($buyQty + $getQty));
            $freeItems = $setsCount * $getQty;

            $totalDiscount += $freeItems * $unitPrice;
        }

        return $totalDiscount;
    }

    protected function calculateSubtotal(Pedido $pedido): float
    {
        return (float) $pedido->detalles->sum('total_usd');
    }

    protected function getExchangeRate(): float
    {
        $rate = ExchangeRate::getLatestRate('USD');

        return $rate ? (float) $rate : 1.0;
    }
}
