<?php

namespace App\Services;

use App\Models\Producto;
use App\Models\InventoryMovement;
use Illuminate\Support\Facades\DB;

class InventoryService
{
    public function recordMovement(
        Producto $producto,
        string $type,
        int $quantity,
        ?float $unitCost = null,
        ?string $reference = null,
        ?string $notes = null
    ): InventoryMovement {
        return DB::transaction(function () use ($producto, $type, $quantity, $unitCost, $reference, $notes) {
            $quantityBefore = $producto->quantity;
            
            // Calcular nueva cantidad según el tipo de movimiento
            $quantityAfter = match($type) {
                'ingreso', 'compra', 'devolucion' => $quantityBefore + $quantity,
                'salida', 'venta' => $quantityBefore - $quantity,
                'ajuste' => $quantity, // Para ajustes, quantity es la cantidad final
                default => $quantityBefore
            };

            // Actualizar cantidad del producto
            $producto->update(['quantity' => $quantityAfter]);

            // Crear registro de movimiento
            return InventoryMovement::create([
                'producto_id' => $producto->id,
                'type' => $type,
                'quantity' => $type === 'ajuste' ? ($quantityAfter - $quantityBefore) : $quantity,
                'quantity_before' => $quantityBefore,
                'quantity_after' => $quantityAfter,
                'unit_cost' => $unitCost,
                'total_cost' => $unitCost ? ($unitCost * abs($quantity)) : null,
                'reference' => $reference,
                'notes' => $notes,
                'user_id' => auth()->id(),
                'empresa_id' => auth()->user()->empresa_id,
                'sucursal_id' => auth()->user()->sucursal_id
            ]);
        });
    }

    public function adjustInventory(Producto $producto, int $newQuantity, string $reason = null): InventoryMovement
    {
        return $this->recordMovement(
            $producto,
            'ajuste',
            $newQuantity,
            null,
            null,
            $reason ?? 'Ajuste de inventario'
        );
    }

    public function getProductMovements(Producto $producto, int $limit = 50)
    {
        return $producto->inventoryMovements()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->paginate($limit);
    }
}