<?php

namespace App\Services;

use Carbon\Carbon;

class CajaTasaConfigService
{
    /**
     * Horarios de corte de caja automático
     */
    const CORTE_CAJA_HORARIOS = [
        '13:00', // 1:00 PM
        '18:00', // 6:00 PM
    ];

    /**
     * Período de verificación de tasa (12:00 AM - 2:00 PM)
     */
    const TASA_VERIFICACION_INICIO = '00:00';
    const TASA_VERIFICACION_FIN = '23:00';

    /**
     * Hora de actualización automática de tasa (2:30 PM)
     */
    const TASA_ACTUALIZACION_AUTO = '14:30';

    /**
     * Verificar si estamos en período de verificación de tasa
     */
    public static function enPeriodoVerificacionTasa(): bool
    {
        $now = Carbon::now();
        $inicio = Carbon::today()->setTimeFromTimeString(self::TASA_VERIFICACION_INICIO);
        $fin = Carbon::today()->setTimeFromTimeString(self::TASA_VERIFICACION_FIN);
        
        return $now->between($inicio, $fin);
    }

    /**
     * Verificar si es hora de actualización automática de tasa
     */
    public static function esHoraActualizacionTasa(): bool
    {
        $now = Carbon::now();
        $horaActualizacion = Carbon::today()->setTimeFromTimeString(self::TASA_ACTUALIZACION_AUTO);
        
        // Considerar un margen de 5 minutos para evitar ejecuciones múltiples
        return $now->between($horaActualizacion, $horaActualizacion->copy()->addMinutes(5));
    }

    /**
     * Verificar si es hora de corte de caja
     */
    public static function esHoraCorteCaja(): ?string
    {
        $now = Carbon::now();
        
        foreach (self::CORTE_CAJA_HORARIOS as $hora) {
            $horaCorte = Carbon::today()->setTimeFromTimeString($hora);
            // Margen de 5 minutos para evitar ejecuciones múltiples
            if ($now->between($horaCorte, $horaCorte->copy()->addMinutes(5))) {
                return $hora;
            }
        }
        
        return null;
    }

    /**
     * Obtener el próximo corte de caja
     */
    public static function getProximoCorteCaja(): ?Carbon
    {
        $now = Carbon::now();
        $hoy = Carbon::today();
        
        foreach (self::CORTE_CAJA_HORARIOS as $hora) {
            $horaCorte = $hoy->copy()->setTimeFromTimeString($hora);
            if ($horaCorte->isFuture()) {
                return $horaCorte;
            }
        }
        
        return null;
    }

    /**
     * Verificar si la tasa está desactualizada (más de 6 horas)
     */
    public static function tasaEstaDesactualizada(?Carbon $ultimaActualizacion): bool
    {
        if (!$ultimaActualizacion) {
            return true;
        }
        
        return $ultimaActualizacion->diffInHours(Carbon::now()) > 6;
    }

    /**
     * Obtener configuración dinámica según hora del día
     */
    public static function getConfiguracionActual(): array
    {
        $now = Carbon::now();
        $config = [
            'hora_actual' => $now->format('H:i'),
            'en_verificacion_tasa' => self::enPeriodoVerificacionTasa(),
            'es_actualizacion_tasa' => self::esHoraActualizacionTasa(),
            'hora_corte_caja' => self::esHoraCorteCaja(),
            'proximo_corte_caja' => self::getProximoCorteCaja()?->format('H:i'),
        ];

        return $config;
    }
}