<?php

namespace App\Services;

use App\Models\ProductBatch;
use App\Models\Producto;

class BatchService
{
    public function allocateStock($productoId, $quantityNeeded, $reference = null)
    {
        $allocations = [];
        $remaining = $quantityNeeded;

        // Obtener lotes disponibles ordenados por FIFO (primero los que vencen antes)
        $batches = ProductBatch::where('producto_id', $productoId)
            ->active()
            ->orderBy('expiry_date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($batches as $batch) {
            if ($remaining <= 0) break;

            $available = min($batch->current_quantity, $remaining);
            
            if ($available > 0) {
                $batch->reduceQuantity($available, 'Sale', $reference);
                
                $allocations[] = [
                    'batch_id' => $batch->id,
                    'batch_number' => $batch->batch_number,
                    'quantity' => $available,
                    'expiry_date' => $batch->expiry_date
                ];
                
                $remaining -= $available;
            }
        }

        return [
            'success' => $remaining == 0,
            'allocated' => $quantityNeeded - $remaining,
            'remaining' => $remaining,
            'batches' => $allocations
        ];
    }

    public function getExpiringProducts($days = 30)
    {
        return ProductBatch::with('producto')
            ->expiringSoon($days)
            ->get()
            ->groupBy('producto_id');
    }

    public function getExpiredProducts()
    {
        return ProductBatch::with('producto')
            ->expired()
            ->get()
            ->groupBy('producto_id');
    }

    public function markExpiredBatches()
    {
        $expiredBatches = ProductBatch::expired()->get();
        
        foreach ($expiredBatches as $batch) {
            $batch->update(['status' => 'expired']);
            
            // Registrar movimiento de expiración
            $batch->movements()->create([
                'type' => 'expired',
                'quantity' => $batch->current_quantity,
                'reason' => 'Producto vencido automáticamente',
                'user_id' => 1, // Sistema
                'empresa_id' => $batch->empresa_id
            ]);
        }
        
        return $expiredBatches->count();
    }
}