<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Tymon\JWTAuth\Contracts\JWTSubject;

class User extends Authenticatable implements MustVerifyEmail, JWTSubject
{
    use HasApiTokens, HasFactory, Notifiable, HasRoles, LogsActivity;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'username',
        'email',
        'telefono',
        'phone',
        'address',
        'bio',
        'password',
        'verification_code',
        'verification_code_sent_at',
        'empresa_id',
        'sucursal_id',
        'status',
        'two_factor_enabled',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'avatar',
        'api_token',
        'notifications_email',
        'notifications_sms',
        'notifications_push',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'verification_code',
        'api_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'two_factor_enabled' => 'boolean',
            'two_factor_recovery_codes' => 'array',
            'verification_code_sent_at' => 'datetime',
            'notifications_email' => 'boolean',
            'notifications_sms' => 'boolean',
            'notifications_push' => 'boolean',
        ];
    }

    /**
     * Get user initials from their name
     */
    public function getInitialsAttribute()
    {
        $names = explode(' ', $this->name);
        $initials = '';
        $count = 0;

        foreach ($names as $name) {
            if ($count < 2) { // Solo tomar las primeras 2 iniciales
                $initials .= strtoupper(substr($name, 0, 1));
                $count++;
            }
        }

        return $initials;
    }

    /**
     * Generar un código de verificación de 8 caracteres (alfanumérico)
     */
    public function generateVerificationCode()
    {
        // Generar código alfanumérico de 8 caracteres
        $code = strtoupper(Str::random(6));

        // Almacenar el código cifrado
        $this->verification_code = Hash::make($code);
        $this->verification_code_sent_at = Carbon::now();
        $this->save();

        // Devolver el código sin cifrar para enviar por correo
        return $code;
    }

    /**
     * Verificar si el código proporcionado es válido
     */
    public function isVerificationCodeValid($code)
    {
        // Verificar que el código exista y no haya expirado (15 minutos)
        if (!$this->verification_code || !$this->verification_code_sent_at) {
            return false;
        }

        if (Carbon::now()->diffInMinutes($this->verification_code_sent_at) > 15) {
            return false;
        }

        // Verificar que el código coincida
        return Hash::check($code, $this->verification_code);
    }

    /**
     * Marcar el correo electrónico como verificado
     */
    public function markEmailAsVerified()
    {
        $this->email_verified_at = Carbon::now();
        $this->verification_code = null;
        $this->verification_code_sent_at = null;
        $this->save();
    }

    /**
     * Get the active sessions for the user.
     */
    public function activeSessions()
    {
        return $this->hasMany(ActiveSession::class);
    }

    /**
     * Get the empresa that owns the user.
     */
    public function empresa()
    {
        return $this->belongsTo(Empresa::class);
    }

    /**
     * Get the sucursal that belongs to the user.
     */
    public function sucursal()
    {
        return $this->belongsTo(Sucursal::class);
    }

    /**
     * Get the user's common locations from active sessions
     */
    public function getCommonLocationsAttribute()
    {
        if (method_exists($this, 'sessions')) {
            return $this->sessions()
                ->whereNotNull('location')
                ->select('location')
                ->distinct()
                ->limit(5)
                ->get()
                ->toArray();
        }

        return [];
    }

    /**
     * Get the messages sent by the user.
     */
    public function mensajesEnviados()
    {
        return $this->hasMany(Mensaje::class, 'remitente_id');
    }

    /**
     * Get the messages received by the user.
     */
    public function mensajesRecibidos()
    {
        return $this->belongsToMany(Mensaje::class, 'mensaje_destinatarios', 'user_id', 'mensaje_id')
            ->withPivot('leido', 'leido_en', 'archivado', 'archivado_en')
            ->withTimestamps()
            ->orderBy('created_at', 'desc');
    }

    /**
     * Get the digital library files uploaded by the user.
     */
    public function archivosBiblioteca()
    {
        return $this->hasMany(BibliotecaArchivo::class, 'usuario_subida_id');
    }

    /**
     * Get the digital library files the user is authorized to access.
     */
    public function archivosBibliotecaAutorizados()
    {
        return $this->belongsToMany(BibliotecaArchivo::class, 'biblioteca_archivo_usuario', 'user_id', 'archivo_id')
            ->withTimestamps();
    }

    /**
     * Get the wishlist items for the user.
     */
    public function wishlist()
    {
        return $this->hasMany(Wishlist::class);
    }

    /**
     * Get the empleado record for the user.
     */
    public function empleado()
    {
        return $this->hasOne(Empleado::class);
    }

    /**
     * Get the pedidos for the user.
     */
    public function pedidos()
    {
        return $this->hasMany(Pedido::class, 'userId');
    }

    /**
     * Get the inventory movements created by the user.
     */
    public function inventoryMovements()
    {
        return $this->hasMany(InventoryMovement::class);
    }

    public function scopeForUser($query)
    {
        if (auth()->check() && !auth()->user()->hasRole('Super Administrador')) {
            if (auth()->user()->empresa_id) {
                $query->where('empresa_id', auth()->user()->empresa_id);
            }
            if (auth()->user()->sucursal_id) {
                $query->where('sucursal_id', auth()->user()->sucursal_id);
            }
        }
        return $query;
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['name', 'email', 'empresa_id', 'sucursal_id', 'status'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    /**
     * Get the identifier that will be stored in the subject claim of the JWT.
     *
     * @return mixed
     */
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    /**
     * Return a key value array, containing any custom claims to be added to the JWT.
     *
     * @return array
     */
    public function getJWTCustomClaims()
    {
        return [];
    }
}
