<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\Tenantable;

class Pedido extends Model
{

    //use Tenantable;
    
    protected $fillable = [
        'fecha',
        'total_usd',
        'total_bs',
        'codigo',
        'metodo_pago',
        'estado',
        'ubicacion',
        'longitud',
        'latitud',
        'nota',
        'empresaId',
        'userId',
        'fecha_entrega'
    ];

    protected $casts = [
        'fecha' => 'datetime',
        'total_usd' => 'decimal:20,2',
        'total_bs' => 'decimal:20,2',
        'latitud' => 'decimal:8',
        'longitud' => 'decimal:8',
        'fecha_entrega' => 'datetime',
    ];

    // Accessors para asegurar que los valores sean numéricos
    public function getTotalUsdAttribute($value)
    {
        return is_numeric($value) ? $value : 0;
    }

    public function getTotalBsAttribute($value)
    {
        return is_numeric($value) ? $value : 0;
    }

    // Scopes
    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    public function scopePagados($query)
    {
        return $query->where('estado', 'pagado');
    }

    public function scopeAsignados($query)
    {
        return $query->where('estado', 'asignado');
    }

    public function scopeEntregados($query)
    {
        return $query->where('estado', 'entregado');
    }

    public function scopeCancelados($query)
    {
        return $query->where('estado', 'cancelado');
    }

    // Accessors
    public function getEstadoBadgeClassAttribute()
    {
        return match($this->estado) {
            'pendiente' => 'bg-warning',
            'pagado' => 'bg-success',
            'asignado' => 'bg-info',
            'entregado' => 'bg-primary',
            'cancelado' => 'bg-danger',
            default => 'bg-secondary'
        };
    }

    public function getEmpleadoAsignadoAttribute()
    {
        return $this->empleados->first();
    }

    /**
     * Get all of the detalles for the Pedido
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function detalles(): HasMany
    {
        return $this->hasMany(DetallePedido::class, 'pedidoId', 'id');
    }

    /**
     * Get all of the detalles with productos for the Pedido
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function detallesConProductos(): HasMany
    {
        return $this->hasMany(DetallePedido::class, 'pedidoId', 'id')->with('producto');
    }


    /**
     * Get the user that owns the Pedido
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'userId', 'id');
    }

    public function empleados()
    {
        return $this->belongsToMany(Empleado::class, 'empleado_pedido', 'pedidoId', 'empleadoId');
    }

   /**
    * Get all of the pagos for the Pedido
    *
    * @return \Illuminate\Database\Eloquent\Relations\HasMany
    */
   public function pagos()
   {
       return $this->hasMany(PedidoPago::class, 'pedidoId', 'id');
   }

   /**
    * Get all of the pagos for the Pedido
    *
    * @return \Illuminate\Database\Eloquent\Relations\HasMany
    */
   public function pago()
   {
       return $this->hasOne(PedidoPago::class, 'id', 'pedidoId');
   }

   /**
    * Get the referencia bancaria for the Pedido
    *
    * @return \Illuminate\Database\Eloquent\Relations\HasOne
    */
   public function referenciaBancaria()
   {
       return $this->hasOne(ReferenciaBancaria::class, 'pedido_id', 'id');
   }

    /**
     * Get the historicos for the Pedido
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function historicos()
    {
        return $this->hasMany(HistoricoPedido::class, 'pedidoId', 'id')->orderBy('created_at', 'desc');
    }

    public function historial()
    {
        return $this->hasMany(PedidoHistorial::class, 'pedido_id')->orderBy('created_at', 'asc');
    }

    /**
     * Get the empresa that owns the Pedido
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function empresa()
    {
        return $this->belongsTo(Empresa::class, 'empresaId');
    }

    /**
     * Get the resenas for the Pedido
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function resenas()
    {
        return $this->hasMany(Resena::class, 'pedido_id', 'id');
    }

    /**
     * Get the discounts for the Pedido
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function discounts(): HasMany
    {
        return $this->hasMany(PedidoDiscount::class, 'pedido_id', 'id');
    }

    /**
     * Accesor para obtener el total formateado en USD
     *
     * @return string
     */
    public function getTotalUsdFormattedAttribute()
    {
        return '$' . number_format($this->total_usd, 2);
    }

    /**
     * Accesor para obtener el total formateado en Bs
     *
     * @return string
     */
    public function getTotalBsFormattedAttribute()
    {
        return number_format($this->total_bs, 2) . ' Bs';
    }

    /**
     * Accesor para obtener la fecha formateada
     *
     * @return string
     */
    public function getFechaFormattedAttribute()
    {
        return $this->fecha ? $this->fecha->format('d/m/Y') : '';
    }

    /**
     * Accesor para obtener la fecha y hora formateada
     *
     * @return string
     */
    public function getFechaHoraFormattedAttribute()
    {
        return $this->fecha ? $this->fecha->format('d/m/Y H:i') : '';
    }

    /**
     * Mutator para establecer la fecha
     *
     * @param  string  $value
     * @return void
     */
    public function setFechaAttribute($value)
    {
        $this->attributes['fecha'] = is_string($value) ? \Carbon\Carbon::createFromFormat('d/m/Y', $value) : $value;
    }

    /**
     * Boot the model.
     *
     * @return void
     */
    protected static function boot()
    {
        parent::boot();

        // Al actualizar el estado a 'entregado', registrar la fecha de entrega
        static::updating(function ($pedido) {
            // Si el estado cambia a 'entregado' y fecha_entrega aún no está establecida
            if ($pedido->isDirty('estado') && $pedido->estado == 'entregado' && !$pedido->fecha_entrega) {
                $pedido->fecha_entrega = now();
            }
        });

        // Disparar evento cuando se actualiza un pedido
        static::updated(function ($pedido) {
            if ($pedido->isDirty('estado')) {
                event(new \App\Events\PedidoUpdated($pedido));
            }
        });

        // Al crear un pedido con estado 'entregado', registrar la fecha de entrega
        static::creating(function ($pedido) {
            if ($pedido->estado == 'entregado' && !$pedido->fecha_entrega) {
                $pedido->fecha_entrega = now();
            }
        });
    }
}