<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class MensajePendiente extends Model
{
    use HasFactory;

    protected $table = 'mensajes_pendientes';

    protected $fillable = [
        'pedido_id',
        'telefono',
        'mensaje',
        'tipo',
        'error',
        'estado',
        'intentos',
        'ultimo_intento',
        'enviado_en'
    ];

    protected $casts = [
        'ultimo_intento' => 'datetime',
        'enviado_en' => 'datetime',
        'intentos' => 'integer'
    ];

    /**
     * Relación con el pedido
     */
    public function pedido()
    {
        return $this->belongsTo(Pedido::class);
    }

    /**
     * Obtener mensajes pendientes por tipo
     */
    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    /**
     * Obtener mensajes por tipo
     */
    public function scopePorTipo($query, string $tipo)
    {
        return $query->where('tipo', $tipo);
    }

    /**
     * Marcar mensaje como enviado
     */
    public function marcarComoEnviado()
    {
        $this->update([
            'estado' => 'enviado',
            'enviado_en' => now()
        ]);
    }

    /**
     * Marcar mensaje como fallido
     */
    public function marcarComoFallido(string $error = null)
    {
        $this->update([
            'estado' => 'fallido',
            'error' => $error,
            'intentos' => $this->intentos + 1,
            'ultimo_intento' => now()
        ]);
    }

    /**
     * Verificar si el mensaje puede ser reenviado
     */
    public function puedeReenviar(): bool
    {
        return $this->estado === 'pendiente' && $this->intentos < 5;
    }

    /**
     * Obtener tipos de mensajes disponibles
     */
    public static function getTipos(): array
    {
        return [
            'asignacion_empleado',
            'asignacion_cliente',
            'confirmacion_pago',
            'cancelacion_pedido',
            'confirmacion_entrega',
            'reversion_pedido'
        ];
    }
}