<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\FiltersByEmpresaSucursal;
use Carbon\Carbon;

class Coupon extends Model
{
    use HasFactory, FiltersByEmpresaSucursal;

    protected $fillable = [
        'promotion_id',
        'empresa_id',
        'code',
        'description',
        'usage_limit',
        'usage_limit_per_user',
        'usage_count',
        'starts_at',
        'ends_at',
        'is_active',
        'metadata',
    ];

    protected $casts = [
        'usage_limit' => 'integer',
        'usage_limit_per_user' => 'integer',
        'usage_count' => 'integer',
        'starts_at' => 'datetime',
        'ends_at' => 'datetime',
        'is_active' => 'boolean',
        'metadata' => 'array',
    ];

    public function promotion(): BelongsTo
    {
        return $this->belongsTo(Promotion::class);
    }

    public function empresa(): BelongsTo
    {
        return $this->belongsTo(Empresa::class);
    }

    public function redemptions(): HasMany
    {
        return $this->hasMany(CouponRedemption::class);
    }

    public function scopeActive($query, $at = null)
    {
        $at = $at ?? Carbon::now();

        return $query->where('is_active', true)
            ->where(function ($q) use ($at) {
                $q->whereNull('starts_at')->orWhere('starts_at', '<=', $at);
            })
            ->where(function ($q) use ($at) {
                $q->whereNull('ends_at')->orWhere('ends_at', '>=', $at);
            });
    }

    public function scopeByCode($query, string $code)
    {
        return $query->where('code', $code);
    }

    public function isValid($at = null): bool
    {
        $at = $at ?? Carbon::now();

        if (!$this->is_active) {
            return false;
        }

        $startsAt = $this->getEffectiveStartsAt();
        $endsAt = $this->getEffectiveEndsAt();

        if ($startsAt && $at->lt($startsAt)) {
            return false;
        }

        if ($endsAt && $at->gt($endsAt)) {
            return false;
        }

        if ($this->usage_limit !== null && $this->usage_count >= $this->usage_limit) {
            return false;
        }

        if ($this->promotion && !$this->promotion->isValid($at)) {
            return false;
        }

        return true;
    }

    public function canBeUsedBy($user): bool
    {
        if (!$this->isValid()) {
            return false;
        }

        if ($this->usage_limit_per_user !== null && $user) {
            $userRedemptions = $this->redemptions()
                ->where('user_id', $user->id)
                ->count();

            if ($userRedemptions >= $this->usage_limit_per_user) {
                return false;
            }
        }

        return true;
    }

    public function getEffectiveStartsAt(): ?Carbon
    {
        if ($this->starts_at) {
            return $this->starts_at;
        }

        return $this->promotion?->starts_at;
    }

    public function getEffectiveEndsAt(): ?Carbon
    {
        if ($this->ends_at) {
            return $this->ends_at;
        }

        return $this->promotion?->ends_at;
    }
}
