<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\Multitenantable;

class Caja extends Model
{
    use HasFactory,Multitenantable;

    protected $fillable = [
        'empresa_id',
        'sucursal_id',
        'user_id',
        'fecha',
        'numero_corte',
        'monto_inicial',
        'total_efectivo',
        'total_transferencias',
        'total_tarjetas',
        'total_ingresos',
        'monto_final',
        'estado',
        'fecha_apertura',
        'fecha_cierre',
        'observaciones_apertura',
        'observaciones_cierre',
    ];

    protected $casts = [
        'fecha' => 'date',
        'fecha_apertura' => 'datetime',
        'fecha_cierre' => 'datetime',
        'monto_inicial' => 'decimal:2',
        'total_efectivo' => 'decimal:2',
        'total_transferencias' => 'decimal:2',
        'total_tarjetas' => 'decimal:2',
        'total_ingresos' => 'decimal:2',
        'monto_final' => 'decimal:2',
    ];

    public function empresa(): BelongsTo
    {
        return $this->belongsTo(Empresa::class);
    }

    public function sucursal(): BelongsTo
    {
        return $this->belongsTo(Sucursal::class);
    }

    public function usuario(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function pedidoPagos(): HasMany
    {
        return $this->hasMany(PedidoPago::class, 'caja_id');
    }

    public function ventas(): HasMany
    {
        return $this->hasMany(Venta::class);
    }

    public function ventaPagos(): HasMany
    {
        return $this->hasMany(VentaPago::class);
    }

    public function movimientos(): HasMany
    {
        return $this->hasMany(CajaMovimiento::class);
    }

    public function calcularTotales(): void
    {
        // Obtener la moneda del país
        $moneda = $this->empresa->pais->moneda_principal ?? 'USD';
        $campoTotal = $moneda === 'VES' ? 'total_bs' : 'total_usd';

        // Primero intentar con PedidoPago (sistema de comercio)
        // Solo sumar pagos de pedidos confirmados (estado = 'pagado' o 'entregado')
        $pedidoPagos = $this->pedidoPagos()
            ->whereHas('pedido', function($query) {
                $query->whereIn('estado', ['pagado', 'entregado']);
            })
            ->where('status', 'Pagado')
            ->get();
        //dd(  $pedidoPagos );
        $totalEfectivo = 0;
        $totalTransferencias = 0;
        $totalTarjetas = 0;

        if ($pedidoPagos->isNotEmpty()) {
            // Sistema de comercio electrónico
            foreach ($pedidoPagos as $pago) {
                $metodo = strtolower(trim($pago->metodo_pago));
                //dd($metodo);
                switch ($metodo) {
                    case 'efectivo':
                    case 'en tienda':
                    case 'efectivo bs':
                    case 'efectivo $':
                        $totalEfectivo += $pago->$campoTotal;
                        break;
                    case 'transferencia':
                    case 'pago_movil':
                    case 'pago móvil':
                        $totalTransferencias += $pago->$campoTotal;
                        break;
                    case 'tarjeta':
                        $totalTarjetas += $pago->$campoTotal;
                        break;
                    case 'punto_de_venta':
                        $totalTarjetas += $pago->$campoTotal;
                        break;
                }
            }
        } else {
            // Fallback al sistema escolar (si existe)
            $pagos = $this->pedidoPagos()
                ->where('status', 'Pagado')
                ->get();

            foreach ($pagos as $pago) {
                if ($pago->es_pago_mixto && $pago->detalles_pago_mixto) {
                    // Procesar pago mixto
                    foreach ($pago->detalles_pago_mixto as $detalle) {
                        switch ($detalle['metodo']) {
                            case 'efectivo_dolares':
                            case 'efectivo_bolivares':
                                $totalEfectivo += $detalle['monto'];
                                break;
                            case 'transferencia':
                            case 'pago_movil':
                                $totalTransferencias += $detalle['monto'];
                                break;
                            case 'tarjeta':
                                $totalTarjetas += $detalle['monto'];
                                break;
                        }
                    }
                } else {
                    // Procesar pago tradicional
                    switch ($pago->metodo_pago) {
                        case 'efectivo':
                        case 'efectivo Bs.':
                        case 'efectivo Divisas.':
                            $totalEfectivo += $pago->total;
                            break;
                        case 'transferencia':
                        case 'pago movil':
                            $totalTransferencias += $pago->total;
                            break;
                        case 'tarjeta':
                            $totalTarjetas += $pago->total;
                            break;
                    }
                }
            }
        }

        $this->total_efectivo = $totalEfectivo;
        $this->total_transferencias = $totalTransferencias;
        $this->total_tarjetas = $totalTarjetas;

        // Agregar ventas POS a los totales (excluyendo ventas canceladas)
        $ventasPOS = $this->ventaPagos()
            ->whereHas('venta', function($query) {
                $query->where('estado', '!=', 'cancelado');
            })
            ->get();
        foreach ($ventasPOS as $ventaPago) {
            switch ($ventaPago->metodo_pago) {
                case 'efectivo':
                    $this->total_efectivo += $ventaPago->monto;
                    break;
                case 'transferencia':
                case 'pago_movil':
                    $this->total_transferencias += $ventaPago->monto;
                    break;
                case 'tarjeta':
                case 'punto_de_venta':
                    $this->total_tarjetas += $ventaPago->monto;
                    break;
            }
        }

        $this->total_ingresos = $this->total_efectivo + $this->total_transferencias + $this->total_tarjetas;

        // Calcular salidas de dinero aprobadas
        $totalSalidas = $this->movimientos()
            ->where('tipo', 'salida')
            ->sum('monto');

        // Calcular monto final considerando salidas
        $this->monto_final = $this->monto_inicial + $this->total_ingresos - $totalSalidas;

        $this->save();
    }

    public function getMontoFinalCalculadoAttribute()
    {
        if ($this->estado === 'cerrada') {
            return $this->monto_final;
        }

        // Calcular salidas de dinero para cajas abiertas
        $totalSalidas = $this->movimientos()
            ->where('tipo', 'salida')
            ->sum('monto');

        return $this->monto_inicial + $this->total_ingresos - $totalSalidas;
    }

    public function cerrar(string $observaciones = null): bool
    {
        if ($this->estado === 'cerrada') {
            return false;
        }

        $this->calcularTotales();
        $this->estado = 'cerrada';
        $this->fecha_cierre = now();
        $this->observaciones_cierre = $observaciones;

        return $this->save();
    }

    public static function obtenerCajaAbierta($empresaId, $sucursalId, $fecha = null): ?self
    {
        $fecha = $fecha ?? now()->toDateString();

        return self::where('empresa_id', $empresaId)
            ->where('sucursal_id', $sucursalId)
            ->where('fecha', $fecha)
            ->where('estado', 'abierta')
            ->first();
    }

    public static function crearCajaDiaria($empresaId, $sucursalId, $montoInicial = 0, $observaciones = null, $userId = null): self
    {
        return self::create([
            'empresa_id' => $empresaId,
            'sucursal_id' => $sucursalId,
            'user_id' => $userId ?? auth()->id() ?? 1,
            'fecha' => now()->toDateString(),
            'numero_corte' => 1,
            'monto_inicial' => $montoInicial,
            'estado' => 'abierta',
            'fecha_apertura' => now(),
            'observaciones_apertura' => $observaciones,
        ]);
    }

    public static function crearCorte($empresaId, $sucursalId, $montoInicial = 0, $observaciones = null, $userId = null): self
    {
        $numeroCorte = self::where('empresa_id', $empresaId)
            ->where('sucursal_id', $sucursalId)
            ->whereDate('fecha', today())
            ->count() + 1;

        return self::create([
            'empresa_id' => $empresaId,
            'sucursal_id' => $sucursalId,
            'user_id' => $userId ?? auth()->id() ?? 1,
            'fecha' => now()->toDateString(),
            'numero_corte' => $numeroCorte,
            'monto_inicial' => $montoInicial,
            'estado' => 'abierta',
            'fecha_apertura' => now(),
            'observaciones_apertura' => ($observaciones ?? '') . " (Corte #{$numeroCorte})",
        ]);
    }
}
