<?php

namespace App\Livewire\Front\Wishlist;

use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Models\Wishlist;
use App\Models\Producto;
use App\Models\Categoria;
use Illuminate\Support\Facades\Auth;
use Cart;

class Index extends Component
{
    use WithPagination;

    public $viewMode = 'grid';
    public $selectedCategory = '';
    public $categories = [];

    public function mount()
    {
        $this->viewMode = session('wishlist_view_mode', 'grid');
        $this->loadCategories();
    }

    public function loadCategories()
    {
        $wishlistProductIds = Wishlist::where('user_id', Auth::id())
            ->pluck('producto_id');
        
        $this->categories = Categoria::whereHas('productos', function($query) use ($wishlistProductIds) {
            $query->whereIn('id', $wishlistProductIds);
        })->withCount(['productos' => function($query) use ($wishlistProductIds) {
            $query->whereIn('id', $wishlistProductIds);
        }])->get()->toArray();
    }

    public function setViewMode($mode)
    {
        $this->viewMode = $mode;
        session(['wishlist_view_mode' => $mode]);
    }

    public function filterByCategory($categoryId)
    {
        $this->selectedCategory = $categoryId === $this->selectedCategory ? '' : $categoryId;
        $this->resetPage();
    }

    public function clearFilters()
    {
        $this->selectedCategory = '';
        $this->resetPage();
    }

    public function addAllToCart()
    {
        try {
            $wishlists = Wishlist::with('producto')
                ->where('user_id', Auth::id())
                ->get();
            
            $addedCount = 0;
            $alreadyInCart = 0;
            
            foreach ($wishlists as $wishlist) {
                $product = $wishlist->producto;
                $cartItem = \Cart::session(userID())->getContent()->where('id', $product->id)->first();
                
                if (!$cartItem) {
                    \Cart::session(userID())->add([
                        'id' => $product->id,
                        'name' => $product->name,
                        'price' => $product->price,
                        'quantity' => 1,
                        'attributes' => [],
                        'associatedModel' => $product
                    ]);
                    $addedCount++;
                } else {
                    $alreadyInCart++;
                }
            }

            $this->dispatch('reloadCartDetail');
            $this->dispatch('cartUpdated');
            
            if ($addedCount > 0) {
                $this->dispatch('alert', [
                    'title' => '¡Productos añadidos!',
                    'text' => "{$addedCount} producto(s) añadido(s) al carrito" . ($alreadyInCart > 0 ? ". {$alreadyInCart} ya estaban en el carrito." : ""),
                    'icon' => 'success',
                ]);
            } else {
                $this->dispatch('alert', [
                    'title' => 'Info',
                    'text' => 'Todos los productos ya están en tu carrito',
                    'icon' => 'info',
                ]);
            }
        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudieron añadir los productos al carrito',
                'icon' => 'error',
            ]);
        }
    }

    public function addToCart($productId)
    {
        try {
            $product = Producto::findOrFail($productId);
            
            $cartItem = \Cart::session(userID())->getContent()->where('id', $productId)->first();
            
            if ($cartItem) {
                $this->dispatch('alert', [
                    'title' => 'Producto ya añadido',
                    'text' => 'Este producto ya está en tu carrito',
                    'icon' => 'info',
                ]);
                return;
            }

            \Cart::session(userID())->add([
                'id' => $product->id,
                'name' => $product->name,
                'price' => $product->price,
                'quantity' => 1,
                'attributes' => [],
                'associatedModel' => $product
            ]);

            $this->dispatch('reloadCartDetail');
            $this->dispatch('cartUpdated');
            
            $this->dispatch('alert', [
                'title' => '¡Producto añadido!',
                'text' => 'El producto ha sido añadido a tu carrito',
                'icon' => 'success',
            ]);
        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo añadir el producto al carrito',
                'icon' => 'error',
            ]);
        }
    }

    #[On('removeFromWishlist')]
    public function removeFromWishlist($wishlistId)
    {
        try {
            $wishlist = Wishlist::where('user_id', Auth::id())->where('id', $wishlistId)->first();
            
            if ($wishlist) {
                $wishlist->delete();
                
                $this->loadCategories();
                
                $this->dispatch('alert', [
                    'title' => '¡Eliminado!',
                    'text' => 'Producto eliminado de tu lista de favoritos',
                    'icon' => 'success',
                ]);
                
                $this->dispatch('reloadWhislist');
            }
        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo eliminar el producto de tu lista de favoritos',
                'icon' => 'error',
            ]);
        }
    }

    public function render()
    {
        $query = Wishlist::with(['producto.categoria', 'producto.marca', 'producto.images'])
            ->where('user_id', Auth::id());
        
        if ($this->selectedCategory) {
            $query->whereHas('producto', function($q) {
                $q->where('categoria_id', $this->selectedCategory);
            });
        }
        
        $wishlists = $query->paginate(12);
            
        return view('livewire.front.wishlist.index', compact('wishlists'))
            ->layout('layouts.front.app');
    }

    #[On('reloadWhislist')]
    public function reload()
    {
        $this->loadCategories();
    }
}
