<?php

namespace App\Livewire\Front\User;

use App\Models\ActiveSession;
use App\Models\Pedido;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;
use Livewire\WithFileUploads;

class Profile extends Component
{
    use WithFileUploads;

    // Tabs
    public $activeTab = 'overview';

    // User data
    public $name;
    public $email;
    public $telefono;
    public $address;
    public $bio;
    public $avatar;
    public $newAvatar;

    // Password change
    public $current_password;
    public $new_password;
    public $new_password_confirmation;

    // Notification preferences
    public $notifications_email = true;
    public $notifications_sms = false;
    public $notifications_push = true;

    // Statistics
    public $totalPedidos = 0;
    public $totalGastado = 0;
    public $pedidosPendientes = 0;
    public $pedidosEntregados = 0;
    public $productosFavoritos = 0;
    public $ultimosPedidos = [];
    public $sesionesActivas = [];
    public $historialSesiones = [];

    protected $rules = [
        'name' => 'required|min:3|max:255',
        'email' => 'required|email',
        'telefono' => 'nullable|string|max:20',
        'address' => 'nullable|string|max:500',
        'bio' => 'nullable|string|max:1000',
        'newAvatar' => 'nullable|image|max:2048',
    ];

    protected $passwordRules = [
        'current_password' => 'required',
        'new_password' => 'required|min:8|confirmed',
    ];

    public function mount()
    {
        $user = Auth::user();
        
        $this->name = $user->name;
        $this->email = $user->email;
        $this->telefono = $user->telefono ?? $user->phone;
        $this->address = $user->address;
        $this->bio = $user->bio;
        $this->avatar = $user->avatar;
        
        $this->notifications_email = $user->notifications_email ?? true;
        $this->notifications_sms = $user->notifications_sms ?? false;
        $this->notifications_push = $user->notifications_push ?? true;

        $this->loadStatistics();
        $this->loadSessions();
    }

    public function loadStatistics()
    {
        $user = Auth::user();
        
        $this->totalPedidos = Pedido::where('userId', $user->id)->count();
        $this->totalGastado = Pedido::where('userId', $user->id)
            ->where('estado', 'Entregado')
            ->sum('total_usd');
        $this->pedidosPendientes = Pedido::where('userId', $user->id)
            ->whereIn('estado', ['Pendiente', 'Procesando', 'Asignado', 'Enviado'])
            ->count();
        $this->pedidosEntregados = Pedido::where('userId', $user->id)
            ->where('estado', 'Entregado')
            ->count();
        $this->productosFavoritos = \App\Models\Wishlist::where('user_id', $user->id)->count();
        
        $this->ultimosPedidos = Pedido::where('userId', $user->id)
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();
    }

    public function loadSessions()
    {
        $user = Auth::user();
        
        $this->sesionesActivas = ActiveSession::where('user_id', $user->id)
            ->where('is_active', true)
            ->orderBy('last_activity', 'desc')
            ->take(5)
            ->get()
            ->toArray();
            
        $this->historialSesiones = ActiveSession::where('user_id', $user->id)
            ->orderBy('login_at', 'desc')
            ->take(20)
            ->get()
            ->toArray();
    }

    public function setTab($tab)
    {
        $this->activeTab = $tab;
    }

    public function updateProfile()
    {
        $this->validate($this->rules);

        $user = Auth::user();
        
        $user->name = $this->name;
        $user->email = $this->email;
        $user->telefono = $this->telefono;
        $user->address = $this->address;
        $user->bio = $this->bio;

        if ($this->newAvatar) {
            $path = $this->newAvatar->store('avatars', 'public');
            $user->avatar = $path;
            $this->avatar = $path;
        }

        $user->save();

        $this->dispatch('alert', [
            'title' => '¡Perfil actualizado!',
            'text' => 'Tu información ha sido actualizada correctamente.',
            'icon' => 'success'
        ]);
    }

    public function updatePassword()
    {
        $this->validate($this->passwordRules);

        $user = Auth::user();

        if (!Hash::check($this->current_password, $user->password)) {
            $this->addError('current_password', 'La contraseña actual no es correcta.');
            return;
        }

        $user->password = Hash::make($this->new_password);
        $user->save();

        $this->reset(['current_password', 'new_password', 'new_password_confirmation']);

        $this->dispatch('alert', [
            'title' => '¡Contraseña actualizada!',
            'text' => 'Tu contraseña ha sido cambiada correctamente.',
            'icon' => 'success'
        ]);
    }

    public function updateNotifications()
    {
        $user = Auth::user();
        
        $user->notifications_email = $this->notifications_email;
        $user->notifications_sms = $this->notifications_sms;
        $user->notifications_push = $this->notifications_push;
        $user->save();

        $this->dispatch('alert', [
            'title' => '¡Preferencias actualizadas!',
            'text' => 'Tus preferencias de notificación han sido guardadas.',
            'icon' => 'success'
        ]);
    }

    public function closeSession($sessionId)
    {
        $session = ActiveSession::find($sessionId);
        
        if ($session && $session->userId === Auth::id()) {
            $session->update([
                'is_active' => false,
                'logout_at' => now()
            ]);
            
            $this->loadSessions();
            
            $this->dispatch('alert', [
                'title' => 'Sesión cerrada',
                'text' => 'La sesión ha sido cerrada correctamente.',
                'icon' => 'success'
            ]);
        }
    }

    public function closeAllOtherSessions()
    {
        $currentSessionId = session()->getId();
        
        ActiveSession::where('userId', Auth::id())
            ->where('session_id', '!=', $currentSessionId)
            ->where('is_active', true)
            ->update([
                'is_active' => false,
                'logout_at' => now()
            ]);
        
        $this->loadSessions();
        
        $this->dispatch('alert', [
            'title' => 'Sesiones cerradas',
            'text' => 'Todas las demás sesiones han sido cerradas.',
            'icon' => 'success'
        ]);
    }

    public function getBrowserIcon($userAgent)
    {
        if (str_contains(strtolower($userAgent), 'chrome')) return 'ri-chrome-line';
        if (str_contains(strtolower($userAgent), 'firefox')) return 'ri-firefox-line';
        if (str_contains(strtolower($userAgent), 'safari')) return 'ri-safari-line';
        if (str_contains(strtolower($userAgent), 'edge')) return 'ri-edge-line';
        return 'ri-global-line';
    }

    public function getDeviceIcon($userAgent)
    {
        if (str_contains(strtolower($userAgent), 'mobile') || str_contains(strtolower($userAgent), 'android')) {
            return 'ri-smartphone-line';
        }
        if (str_contains(strtolower($userAgent), 'tablet') || str_contains(strtolower($userAgent), 'ipad')) {
            return 'ri-tablet-line';
        }
        return 'ri-computer-line';
    }

    public function render()
    {
        return view('livewire.front.user.profile')->layout('layouts.front.app');
    }
}
