<?php

namespace App\Livewire\Front;

use App\Models\Producto;
use App\Models\Wishlist;
use Livewire\Component;
use Livewire\Attributes\On;

class QuickView extends Component
{
    public $product = null;
    public $relatedProducts = [];
    public $showModal = false;
    public $selectedVariant = null;
    public $variantOptions = [];
    public $cartItemId = null; // ID del item del carrito si estamos actualizando
    public $currency = ''; // Default currency
    public $fav = []; // Wishlist items for the authenticated user
    public $quantity = 1; // Quantity selector




   function mount()
   {
        // Restaurar preferencias del usuario desde la sesión
        if (session()->has('currency')) {
            $this->currency = session('currency');
        }
   }


    #[On('openQuickView')]
    public function openModal($productId, $cartItemId = null)
    {
        $this->product = Producto::with(['categoria', 'marca', 'images', 'variants'])->find($productId);
        $this->cartItemId = $cartItemId;

        // Set initial currency based on system configuration
        $this->setInitialCurrency();

        // Load user's wishlist items
        $this->loadWishlist();

        // Debug: Ver todas las variantes con sus imágenes
        if ($this->product && $this->product->variants->count() > 0) {
            \Log::info('Variantes cargadas en quickview', [
                'product_id' => $productId,
                'variants' => $this->product->variants->map(function($v) {
                    return [
                        'id' => $v->id,
                        'name' => $v->name,
                        'image_path' => $v->image_path,
                        'image_thumbnail_path' => $v->image_thumbnail_path
                    ];
                })->toArray()
            ]);

            // No seleccionar ninguna variante por defecto
            $this->selectedVariant = null;
        }

        $this->quantity = 1; // Reset quantity
        $this->loadRelatedProducts();
        $this->showModal = true;
    }

    // Set initial currency based on system configuration
    private function setInitialCurrency()
    {
        // Use session currency if available, otherwise default to USD
        $this->currency = session('currency', 'usd');
    }

    #[On('currencyChanged')]
    public function updateCurrency($currency)
    {
        $this->currency = $currency;
        session(['currency' => $currency]);
    }

    // Load user's wishlist items
    private function loadWishlist()
    {
        if (auth()->check()) {
            $this->fav = Wishlist::where('user_id', auth()->id())->get();
        } else {
            $this->fav = collect(); // Empty collection for guest users
        }
    }

    public function loadRelatedProducts()
    {
        if ($this->product) {
            $this->relatedProducts = Producto::with(['categoria', 'marca', 'images'])
                ->where('categoria_id', $this->product->categoria_id)
                ->where('id', '!=', $this->product->id)
                ->where('status', 1)
                ->take(12)
                ->get();
        }
    }

    public function selectRelatedProduct($productId)
    {
        $this->product = Producto::with(['categoria', 'marca', 'images', 'variants'])->find($productId);

        // No seleccionar ninguna variante por defecto
        $this->selectedVariant = null;

        $this->loadRelatedProducts();

        // Forzar re-renderizado completo del componente
        $this->dispatch('refreshQuickView');

        // Emitir evento para actualizar la imagen con transición suave
        $this->dispatch('productChanged', [
            'product_id' => $productId,
            'has_images' => $this->product->images && $this->product->images->isNotEmpty()
        ]);

        \Log::info('Producto relacionado seleccionado', [
            'product_id' => $productId,
            'product_name' => $this->product->name,
            'image_count' => $this->product->images ? $this->product->images->count() : 0
        ]);
    }

    public function selectVariant($variantId)
    {
        \Log::info('selectVariant method called', ['variant_id' => $variantId]);

        $this->selectedVariant = $this->product->variants->find($variantId);

        \Log::info('Variante seleccionada en quickview', [
            'variant_id' => $variantId,
            'image_path' => $this->selectedVariant->image_path ?? 'null',
            'image_thumbnail_path' => $this->selectedVariant->image_thumbnail_path ?? 'null'
        ]);

        // Forzar re-renderizado completo del componente
        $this->dispatch('refreshQuickView');

        // Emitir evento para actualizar la imagen con transición suave
        $this->dispatch('variantSelected', [
            'variant_id' => $variantId,
            'has_image' => !empty($this->selectedVariant->image_path)
        ]);
        \Log::info('variantSelected event dispatched');
    }

    // Method to switch currency
    public function switchCurrency($currency)
    {
        $this->currency = $currency;
    }

    // Increment quantity
    public function incrementQuantity()
    {
        $stock = $this->selectedVariant->stock ?? $this->product->stock ?? 100;
        if ($this->quantity < $stock) {
            $this->quantity++;
        }
    }

    // Decrement quantity
    public function decrementQuantity()
    {
        if ($this->quantity > 1) {
            $this->quantity--;
        }
    }

    // Add product to wishlist
    public function addWishlist($productId)
    {
        if (!auth()->check()) {
            $this->dispatch('alert', [
                'text' => 'Debes iniciar sesión para agregar productos a tu lista de deseos',
                'icon' => 'warning',
            ]);
            return;
        }

        $wishlistItem = Wishlist::where('user_id', auth()->id())
            ->where('producto_id', $productId)
            ->first();

        if ($wishlistItem) {
            $wishlistItem->delete();
            $this->dispatch('alert', [
                'text' => 'Producto eliminado de tu lista de deseos',
                'icon' => 'info',
            ]);
        } else {
            Wishlist::create([
                'user_id' => auth()->id(),
                'producto_id' => $productId,
            ]);
            $this->dispatch('alert', [
                'text' => 'Producto agregado a tu lista de deseos',
                'icon' => 'success',
            ]);
        }

        // Reload wishlist
        $this->loadWishlist();
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->product = null;
        $this->selectedVariant = null;
        $this->relatedProducts = [];

        // Emitir evento para reiniciar el carrusel principal cuando se cierre el modal
        $this->dispatch('quickViewModalClosed');
    }

    public function addToCart()
    {
        if ($this->product) {
            $productToAdd = $this->selectedVariant ?? $this->product;

            // Calcular precio según la moneda seleccionada
            if ($this->currency === 'bs') {

                if ($this->selectedVariant ?? false) {
                    // Para variantes, usar precio_bs de la variante si existe, sino calcular
                    $price = $this->selectedVariant->precio_bs *  \App\Models\ExchangeRate::getLatestRate('USD');

                }
                else
                    {
                         $price = $this->product->precio_bs *  \App\Models\ExchangeRate::getLatestRate('USD');
                    }
            } else {
                // Para USD, usar precio base + ajuste de variante
              if ($this->selectedVariant) {
                    // Para variantes, usar precio_bs de la variante si existe, sino calcular
                    $price = $this->selectedVariant->price_adjustment;

                }
                else
                    {
                         $price = $this->product->price;
                    }
            }

            // Si estamos actualizando un item existente del carrito
            if ($this->cartItemId) {
                // Obtener la cantidad actual del item antes de eliminarlo
                $currentItem = \Cart::session(userID())->get($this->cartItemId);
                $currentQuantity = $currentItem ? $currentItem->quantity : 1;

                if (!$currentItem) {
                    \Log::warning('Cart item not found for update', ['cartItemId' => $this->cartItemId]);
                }

                // Primero eliminamos el item actual
                \Cart::session(userID())->remove($this->cartItemId);

                // Luego agregamos el nuevo con la variante seleccionada y la misma cantidad
                $newItemId = $productToAdd->id . '_' . ($this->selectedVariant ? $this->selectedVariant->id : '0');

                // Procesar valores de variante



                if ($this->selectedVariant) {
                    $values = is_string($this->selectedVariant->values) ? json_decode($this->selectedVariant->values, true) : $this->selectedVariant->values;
                    if (is_array($values)) {
                        // Si es el formato {"Peso": "71G"}, extraer solo los valores
                        $displayValues = [];
                        foreach ($values as $key => $value) {
                            $displayValues[] = $key . ': ' . $value;
                        }
                        $variantName = implode(', ', $displayValues);
                    }
                }



                \Cart::session(userID())->add([
                    'id' => $newItemId,
                    'name' => $variantName,
                    'price' => $price,
                    'quantity' => $currentQuantity,
                    'attributes' => [
                        'variant_id' => $this->selectedVariant ? $this->selectedVariant->id : null,
                        'currency' => $this->currency
                    ],
                    'associatedModel' => $this->product
                ]);

                \Log::info('Cart item updated', [
                    'old_cart_item_id' => $this->cartItemId,
                    'new_item_id' => $newItemId,
                    'product_id' => $productToAdd->id,
                    'variant_id' => $this->selectedVariant ? $this->selectedVariant->id : null,
                    'quantity' => $currentQuantity,
                    'price' => $price,
                    'currency' => $this->currency
                ]);

                $this->dispatch('alert', [
                    'title' => '¡Actualizado!',
                    'text' => '¡Producto actualizado en el carrito!',
                    'icon' => 'success',
                ]);
            } else {
                // Agregar nuevo producto al carrito
                $newItemId = $productToAdd->id . '_' . ($this->selectedVariant ? $this->selectedVariant->id : '0');

                // Procesar valores de variante
                $variantName = '';
                if ($this->selectedVariant) {
                    $values = is_string($this->selectedVariant->values) ? json_decode($this->selectedVariant->values, true) : $this->selectedVariant->values;
                    if (is_array($values)) {
                        // Si es el formato {"Peso": "71G"}, extraer solo los valores
                        $displayValues = [];
                        foreach ($values as $key => $value) {
                            if (is_string($value)) {
                                $displayValues[] = $value;
                            }
                        }
                        if (!empty($displayValues)) {
                            $variantName = ' - ' . implode(', ', $displayValues);
                        }
                    }
                }

                \Cart::session(userID())->add([
                    'id' => $newItemId,
                    'name' => preg_replace('/\s+\S+$/', '', $this->product->name).''.$variantName,
                    'price' => $price,
                    'quantity' => $this->quantity,
                    'attributes' => [
                        'variant_id' => $this->selectedVariant ? $this->selectedVariant->id : null,
                        'currency' => $this->currency
                    ],
                    'associatedModel' => $this->product
                ]);

                \Log::info('New cart item added', [
                    'item_id' => $newItemId,
                    'product_id' => $productToAdd->id,
                    'variant_id' => $this->selectedVariant ? $this->selectedVariant->id : null,
                    'quantity' => $this->quantity,
                    'price' => $price,
                    'currency' => $this->currency
                ]);

                $this->dispatch('alert', [
                    'title' => '¡Bien hecho!',
                    'text' => '¡Producto agregado al carrito!',
                    'icon' => 'success',
                ]);
            }

            // Forzar actualización completa del carrito
            $this->dispatch('cartUpdated');
            $this->dispatch('reloadCartDetail');
            $this->closeModal();
        }
    }

    public function render()
    {
        return view('livewire.front.quick-view');
    }
}
