<?php

namespace App\Livewire\Front;

use App\Models\Producto;
use App\Models\User;
use App\Models\Wishlist;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Validation\Rules\Password;
use Illuminate\Validation\ValidationException;
use App\Services\AIRecommendationService;

class Index extends Component
{

    use WithPagination;

    protected $rules = [
        'username' => 'required',
        'password' => 'required',
    ];

    protected $paginationTheme = 'bootstrap';

    public $categorias = [];
    public $marcas = [];
    public $wishlists  = [];
    public $fav = [];
    public $perPage    = 12;
    public $search     = '';
    public $categoriaId;
    public $marcaId;
    public $minPrice;
    public $maxPrice;
    public $sortBy = 'id';
    public $sortDirection = 'desc';
    public $viewMode = 'grid';
    public $currency = 'usd'; // usd or bs
    public $isRegistering = false;
    public $cartCount = 0;
    public $aiRecommendations = [];
    public $bestSellingProducts = [];
    public $latestCustomers = [];
    public
        $name,
        $username,
        $email,
        $telefono,
        $password,
        $password_confirmation,
        $forgotEmail,
        $resetMessage,
        $resetError;



    public $start_date;
    public $end_date;

    protected $queryString = [
        'search' => ['except' => ''],
        'categoriaId' => ['except' => ''],
        'marcaId' => ['except' => ''],
        'sortBy' => ['except' => 'id'],
        'sortDirection' => ['except' => 'desc'],
        'viewMode' => ['except' => 'grid'],
    ];

    public function mount()
    {
        $this->categorias = $this->getCategoriasWithCount();
        $this->marcas = $this->getMarcasWithCount();
        $this->fav = \App\Models\Wishlist::where('user_id', \Auth::id())->get();
        $this->start_date = today()->subDays(30)->format('Y-m-d');
        $this->end_date   = today()->addDays(1)->format('Y-m-d');
        $this->updateCartCount();
        $this->loadAIRecommendations();
        $this->loadBestSellingProducts();
        $this->loadLatestCustomers();

        // Restaurar preferencias del usuario desde la sesión
        if (session()->has('currency')) {
            $this->currency = session('currency');
        }

        if (session()->has('viewMode')) {
            $this->viewMode = session('viewMode');
        }
    }

    // Método para obtener sugerencias de búsqueda mejorado
    public function getSuggestionsProperty()
    {
        if (strlen($this->search) < 2) {
            return collect();
        }

        return Cache::remember('search_suggestions_' . md5($this->search), 300, function () {
            // Búsqueda en productos, categorías y marcas
            $productos = \App\Models\Producto::where('status', 1)
                ->where('name', 'LIKE', '%' . $this->search . '%')
                ->limit(5)
                ->get()
                ->map(fn($p) => ['type' => 'producto', 'name' => $p->name, 'id' => $p->id]);

            $categorias = \App\Models\Categoria::where('activo', 1)
                ->where('nombre', 'LIKE', '%' . $this->search . '%')
                ->limit(3)
                ->get()
                ->map(fn($c) => ['type' => 'categoria', 'name' => $c->nombre, 'id' => $c->id]);

            return $productos->concat($categorias);
        });
    }

    // Obtener productos relacionados
    public function getRelatedProducts($productId, $limit = 4)
    {
        $product = \App\Models\Producto::find($productId);
        if (!$product) return collect();

        return \App\Models\Producto::where('status', 1)
            ->where('id', '!=', $productId)
            ->where(function($query) use ($product) {
                $query->where('categoria_id', $product->categoria_id)
                      ->orWhere('marca_id', $product->marca_id);
            })
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }

    // Método para agregar productos al carrito
    function addCart($id)
    {
        try {
            $producto = \App\Models\Producto::find($id);

            if (!$producto) {
                $this->dispatch('alert', [
                    'title' => 'Producto no encontrado',
                    'text'  => 'El producto solicitado no existe',
                    'icon'  => 'error',
                ]);
                return;
            }

            // Verificar si el producto tiene stock
            if ($producto->quantity <= 0) {
                $this->dispatch('alert', [
                    'title' => 'Producto no disponible',
                    'text'  => 'Este producto está agotado',
                    'icon'  => 'warning',
                ]);
                return;
            }

            // Verificar si el precio en la moneda seleccionada está disponible
            if ($this->currency === 'bs' && !$producto->precio_bs) {
                $this->dispatch('alert', [
                    'title' => 'Precio no disponible',
                    'text'  => 'Este producto no tiene precio en bolívares',
                    'icon'  => 'warning',
                ]);
                return;
            }

            $exchangeRate = \App\Models\ExchangeRate::getLatestRate('USD') ?: 36;
            // Determinar el precio según la moneda seleccionada
            $price = $this->currency === 'bs' ? $producto->precio_bs * $exchangeRate  : $producto->price;

            // Verificar si el producto ya está en el carrito
            $cartItem = \Cart::session(userID())->get($id);

            if ($cartItem) {
                // Verificar stock antes de incrementar
                if ($cartItem->quantity >= $producto->quantity) {
                    $this->dispatch('alert', [
                        'title' => 'Stock insuficiente',
                        'text' => 'No hay más unidades disponibles de este producto',
                        'icon' => 'warning',
                    ]);
                    return;
                }

                // Incrementar cantidad
                \Cart::session(userID())->update($id, [
                    'quantity' => 1
                ]);
            } else {
                // Agregar nuevo producto al carrito
                \Cart::session(userID())->add(array(
                    'id' => $producto->id,
                    'name' => $producto->name,
                    'price' => $price,
                    'quantity' => 1,
                    'attributes' => array('currency' => $this->currency),
                    'associatedModel' => $producto
                ));
            }

            $this->dispatch('alert', [
                'title' => '¡Bien hecho!',
                'text'  => '¡Producto agregado al carrito!',
                'icon'  => 'success',
            ]);

            // Actualizar contador y emitir eventos
            $this->updateCartCount();
            $this->dispatch('reloadCartDetail');
            $this->dispatch('cartUpdated');
            $this->dispatch('contentChanged');
        } catch (\Exception $e) {
            \Log::error('Error adding product to cart', [
                'product_id' => $id,
                'user_id' => auth()->id(),
                'error' => $e->getMessage()
            ]);

            $this->dispatch('alert', [
                'title' => 'Error',
                'text'  => 'Hubo un problema al agregar el producto al carrito',
                'icon'  => 'error',
            ]);
        }
    }

    public function render()
    {
        // Usar caché para productos recientes
        $recent_products = Cache::remember('recent_products', 300, function() {
            return \App\Models\Producto::with(['categoria', 'marca', 'images'])
                ->where('status', 1)
                ->orderBy('id', 'desc')
                ->take(12)
                ->get();
        });

        // Usar caché para productos más vendidos
        $best_selling_qty = Cache::remember('best_selling_products_' . $this->start_date . '_' . $this->end_date, 1800, function() {
            return \App\Models\DetallePedido::where('empresaId', 1)
                ->whereDate('created_at', '>=', $this->start_date)
                ->whereDate('created_at', '<=', $this->end_date)
                ->select(\DB::raw('productoId, sum(quantity) as cantidad'))
                ->whereMonth('created_at', '<=', date('m'))
                ->groupBy('productoId')
                ->orderBy('cantidad', 'desc')
                ->take(10)
                ->get();
        });

        // Contar productos totales
        $total_products = Cache::remember('total_active_products', 600, function() {
            return \App\Models\Producto::where('status', 1)->count();
        });

        // Contar clientes
        $total_customers = Cache::remember('total_customers', 600, function() {
            return \App\Models\User::role('Cliente')->count();
        });

        return view('livewire.front.index', [
            'productos' => Producto::with(['categoria', 'marca', 'images'])
                ->where('status', 1)
                ->when($this->search, function ($query) {
                    return $query->where('name', 'LIKE', '%' . $this->search . '%')
                        ->orWhereHas('categoria', function($q) {
                            $q->where('nombre', 'LIKE', '%' . $this->search . '%');
                        })
                        ->orWhereHas('marca', function($q) {
                            $q->where('nombre', 'LIKE', '%' . $this->search . '%');
                        });
                })
                ->when($this->categoriaId, function ($query) {
                    return $query->where('categoria_id', $this->categoriaId);
                })
                ->when($this->marcaId, function ($query) {
                    return $query->where('marca_id', $this->marcaId);
                })
                ->when($this->minPrice, function ($query) {
                    return $query->where('price', '>=', $this->minPrice);
                })
                ->when($this->maxPrice, function ($query) {
                    return $query->where('price', '<=', $this->maxPrice);
                })
                ->orderBy($this->sortBy, $this->sortDirection)
                ->paginate($this->perPage),
            'recent_products' => $recent_products,
            'best_selling_qty' => $best_selling_qty,
            'categorias' => $this->categorias,
            'marcas' => $this->marcas,
            'total_products' => $total_products,
            'total_customers' => $total_customers,
            'suggestions' => $this->suggestions
        ])
            ->layout('layouts.front.app');
    }

    function addWishlist($id)
    {
        // Verificar si el producto ya está en la lista de favoritos
        $existingWishlistItem = Wishlist::where('producto_id', $id)
            ->where('user_id', \Auth::id())
            ->first();

        if ($existingWishlistItem != null) {
            // Si el producto ya está en favoritos, lo eliminamos
            $existingWishlistItem->delete();

            // Actualizar la lista de favoritos para que el icono cambie inmediatamente
            $this->fav = Wishlist::where('user_id', \Auth::id())->get();

            $this->dispatch('reloadWhislist');
            $this->dispatch('contentChanged');

            $this->dispatch('alert', [
                'title' => '¡Bien hecho!',
                'text'  => 'Producto eliminado de favoritos!',
                'icon'  => 'success',
            ]);
        } else {
            // Si el producto no está en favoritos, lo añadimos
            $wishlist = new Wishlist();
            $wishlist->producto_id = $id;
            $wishlist->user_id     = \Auth::id();
            $wishlist->save();

            // Actualizar la lista de favoritos para que el icono cambie inmediatamente
            $this->fav = Wishlist::where('user_id', \Auth::id())->get();

            $this->dispatch('reloadWhislist');
            $this->dispatch('contentChanged');

            $this->dispatch('alert', [
                'title' => '¡Bien hecho!',
                'text'  => 'Producto añadido a favoritos!',
                'icon'  => 'success',
            ]);
        }
    }

    function removeWishlist($id)
    {
        $wishlist = Wishlist::where('producto_id', $id)
            ->where('user_id', \Auth::id())
            ->first();

        if ($wishlist) {
            $wishlist->delete();

            // Actualizar la lista de favoritos para que el icono cambie inmediatamente
            $this->fav = Wishlist::where('user_id', \Auth::id())->get();

            $this->dispatch('reloadWhislist');

            $this->dispatch('alert', [
                'title' => '¡Bien hecho!',
                'text'  => 'Datos registrados satisfactoriamente!',
                'icon'  => 'success',
            ]);
        }
    }

    function valueCategoria_id($value)
    {
        $this->categoria_id = $value;
    }

    public function setCurrency($currency)
    {
        $this->currency = $currency;
        session(['currency' => $currency]);
        $this->dispatch('currencyChanged', $currency);
    }

    function storeUser()
    {
        $this->isRegistering = true;

        $this->validate([
            'name' => 'required',
            'email' => 'required|email|unique:users',
            'username' => 'required|unique:users',
            'telefono' => ['required', 'unique:users', function($attribute, $value, $fail) {
                if (!$this->isValidPhoneNumber($value)) {
                    $fail('El número de teléfono no es válido para el país.');
                }
            }],
            'password' => [
                'required',
                'max:50',
                Password::min(8),
            ],
            "password_confirmation" => "required|min:8|max:50|same:password",
        ]);

        try {
            DB::beginTransaction();
            $user = new User();
            $user->name = $this->name;
            $user->email      = $this->email;
            $user->username   = $this->username;
            $user->telefono   = $this->formatPhoneNumber($this->telefono);
            $user->password   = $this->password;
            $user->empresa_id = 1;
            $user->sucursal_id = 1;
            $user->email_verified_at = now();
            $user->status     = 1;
            $user->save();
            $user->assignRole('Cliente');
            DB::commit();

            Auth::loginUsingId($user->id);

            // Generar token temporal para credenciales
            $credentialToken = \App\Services\TemporaryTokenService::generateCredentialToken(
                $user->id, $user->username, $this->password
            );

            // Enviar mensaje de bienvenida sincrónicamente
            try {
                $whatsappService = new \App\Services\WhatsAppService();
                $telefonoFormateado = $whatsappService->cleanPhoneNumber($user->telefono);

                $message = "🎉 *¡Bienvenido {$user->name}!*\n\n";
                $message .= "Gracias por registrarte en nuestro sistema de pedidos.\n\n";
                $message .= "📱 *Tus credenciales de acceso:*\n";
                $message .= "Usuario: {$user->username}\n";
                $message .= "Contraseña: {$this->password}\n";
                $message .= "Ingresa a nuestro sistema y realiza tus pedidos de manera fácil y rápida.\n\n";
                $message .= "¿Necesitas ayuda? No dudes en contactarnos.\n\n";
                $message .= "¡Bienvenido a bordo! 🚀";

                $result = $whatsappService->sendMessage($telefonoFormateado, $message);

                if ($result && ($result['success'] || isset($result['simulated']))) {
                    \Log::info('Mensaje de bienvenida enviado exitosamente', [
                        'user_id' => $user->id,
                        'telefono' => $telefonoFormateado
                    ]);
                } else {
                    \Log::warning('No se pudo enviar mensaje de bienvenida', [
                        'user_id' => $user->id,
                        'telefono' => $telefonoFormateado,
                        'result' => $result
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Error al enviar mensaje de bienvenida: ' . $e->getMessage(), [
                    'user_id' => $user->id,
                    'telefono' => $user->telefono
                ]);
            }

            $this->isRegistering = false;

            $this->dispatch('alert', [
                'title' => '¡Bien hecho!',
                'text'  => '¡Cliente registrado satisfactoriamente!',
                'icon'  => 'success',
            ]);
            return redirect('/front/store');
        } catch (\Throwable $th) {
            DB::rollback();
            $this->isRegistering = false;

            $this->dispatch('alert', [
                'title' => 'Error',
                'text'  => '¡Algo salió mal al enviar el formulario!',
                'icon'  => 'warning',
            ]);
        }
    }

    function loginUser()
    {
        $this->validate([
            'username' => 'required',
            'password' => [
                'required',
                'max:50',
                Password::min(8),
            ],
        ]);

        $attributes = [
            'username' => $this->username,
            'password' => $this->password,
        ];

        if (! Auth::attempt($attributes)) {
            throw ValidationException::withMessages([
                'username' => 'Usuario o contraseña incorrectos.'
            ]);
        }

        $this->dispatch('alert', [
            'title' => '¡Bien hecho!',
            'text'  => '¡Sesión iniciada!',
            'icon'  => 'success',
        ]);
        return redirect('/front/store');
    }

    function filterCategoria($value)
    {
        $this->categoria_id = $value;
    }

    public function selectSuggestion($name)
    {
        $this->search = $name;
        $this->resetPage();
        $this->dispatch('searchSelected');
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedCategoriaId()
    {
        $this->resetPage();
        $this->dispatch('contentChanged');
    }
    public function updatedCurrency($currency)
    {
         if (session()->has('currency')) {
            session()->forget(['currency', 'other_key']);
          }
        $this->currency = $currency;
        session(['currency' => $currency]);
        $this->dispatch('currencyChanged', $currency);
    }

    public function sortBy($field)
    {
        if ($this->sortBy === $field) {
            // Si estamos ordenando por el mismo campo, invertimos la dirección
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            // Si es un campo diferente, establecemos la dirección predeterminada
            $this->sortBy = $field;
            // Para nombre, ascendente por defecto. Para precio y fecha, descendente por defecto
            $this->sortDirection = ($field === 'name') ? 'asc' : 'desc';
        }

        $this->resetPage();
        $this->dispatch('$refresh');
        $this->dispatch('contentChanged');
    }

    public function setViewMode($mode)
    {
        $this->viewMode = $mode;
        session(['viewMode' => $mode]);
        $this->dispatch('contentChanged');
    }

    public function clearFilters()
    {
        $this->reset(['search', 'categoriaId', 'marcaId', 'minPrice', 'maxPrice']);
        $this->resetPage();
        $this->dispatch('contentChanged');
    }

    public function quickView($productId)
    {
        $this->dispatch('openQuickView', productId: $productId);
    }

    public function addToComparison($productId)
    {
        $this->dispatch('productAdded', productId: $productId);
    }

    private function loadAIRecommendations()
    {
        if (auth()->check()) {
            $this->aiRecommendations = Cache::remember('ai_recommendations_user_' . auth()->id(), 3600, function() {
                $aiService = new AIRecommendationService();
                return $aiService->getRecommendations(auth()->id(), 6);
            });
        }
    }

    private function loadBestSellingProducts()
    {
        $this->bestSellingProducts = Cache::remember('best_selling_products', 3600, function() {
            return \App\Models\Producto::with(['categoria', 'marca', 'images'])
                ->where('status', 1)
                ->whereExists(function($query) {
                    $query->select(DB::raw(1))
                        ->from('detalle_pedidos')
                        ->whereRaw('detalle_pedidos.productoId = productos.id')
                        ->whereExists(function($subquery) {
                            $subquery->select(DB::raw(1))
                                ->from('pedidos')
                                ->whereRaw('pedidos.id = detalle_pedidos.pedidoId')
                                ->where('pedidos.estado', 'entregado');
                        });
                })
                ->selectRaw('productos.*, (
                    SELECT SUM(quantity)
                    FROM detalle_pedidos
                    WHERE productos.id = detalle_pedidos.productoId
                    AND EXISTS (
                        SELECT 1 FROM pedidos
                        WHERE detalle_pedidos.pedidoId = pedidos.id
                        AND pedidos.estado = "entregado"
                    )
                ) as total_sales')
                ->orderBy('total_sales', 'desc')
                ->limit(12)
                ->get();
        });
    }

    private function loadLatestCustomers()
    {
        $this->latestCustomers = Cache::remember('latest_customers', 1800, function() {
            return \App\Models\User::role('Cliente')
                ->where('status', 1)
                ->latest()
                ->limit(12)
                ->get();
        });
    }


    function logout()
    {
        Auth::logout();
        return redirect('front/store');
    }

    public function sendPasswordResetCode()
    {
        $this->validate([
            'forgotEmail' => 'required|exists:users,telefono'
        ], [
            'forgotEmail.required' => 'El número de teléfono es obligatorio.',
            'forgotEmail.exists' => 'No encontramos una cuenta con este número de teléfono.'
        ]);

        try {
            $user = \App\Models\User::where('telefono', $this->forgotEmail)->first();

            // Generar código de 8 dígitos
            $newPassword = str_pad(random_int(10000000, 99999999), 8, '0', STR_PAD_LEFT);

            // Actualizar contraseña del usuario
            $user->update(['password' => bcrypt($newPassword)]);

            // Enviar código por WhatsApp directamente
            $whatsappService = new \App\Services\WhatsAppService();
            $telefonoFormateado = $whatsappService->cleanPhoneNumber($user->telefono);

            $message = "🔐 *Nueva Contraseña*\n\n";
            $message .= "Hola {$user->name}!\n\n";
            $message .= "Tu nueva contraseña es: *{$newPassword}*\n\n";
            $message .= "Por seguridad, te recomendamos cambiarla después de iniciar sesión.\n\n";
            $message .= "¡Gracias por usar nuestros servicios!";

            $result = $whatsappService->sendMessage($telefonoFormateado, $message);

            if ($result['success'] || isset($result['simulated'])) {
                $this->resetMessage = 'Te hemos enviado tu nueva contraseña por WhatsApp.';
                $this->resetError = '';
                $this->forgotEmail = '';
            } else {
                // Revertir cambio de contraseña si falla el WhatsApp
                $user->update(['password' => $user->getOriginal('password')]);
                $this->resetError = 'Error al enviar el mensaje. Inténtalo de nuevo.';
                $this->resetMessage = '';
            }
        } catch (\Exception $e) {
            $this->resetError = 'Hubo un problema al procesar tu solicitud.';
            $this->resetMessage = '';
        }
    }

    function wishList($value)
    {
        $this->wishlists = \App\Models\Wishlist::where('user_id', $value)->get();
    }

    private function formatPhoneNumber($phoneNumber)
    {
        // Obtener empresa activa y su país
        $empresa = \App\Models\Empresa::where('status', 1)->first();
        if (!$empresa || !$empresa->pais) {
            return $phoneNumber; // Retornar sin cambios si no hay empresa o país
        }

        $pais = $empresa->pais;
        $codigoPais = $pais->codigo_telefonico ?? '+58'; // Default Venezuela

        // Remover el símbolo + del código de país para comparaciones
        $codigoLimpio = $codigoPais;

        // Limpiar el número (remover espacios, guiones, paréntesis, +)
        $cleanNumber = preg_replace('/[\s\-\(\)\+]/', '', $phoneNumber);

        // Si el número ya tiene el código de país, devolverlo
        if (substr($cleanNumber, 0, strlen($codigoLimpio)) === $codigoLimpio) {
            return $cleanNumber;
        }

        // Si empieza con 0, removerlo (formato local)
        if (substr($cleanNumber, 0, 1) === '0') {
            $cleanNumber = substr($cleanNumber, 1);
        }

        // Agregar código de país (sin el símbolo +)
        return $codigoLimpio . $cleanNumber;
    }

    private function isValidPhoneNumber($phoneNumber): bool
    {
        $empresa = \App\Models\Empresa::where('status', 1)->first();
        if (!$empresa || !$empresa->pais) {
            return true; // Skip validation if no country config
        }

        $cleanNumber = preg_replace('/[\s\-\(\)\+]/', '', $phoneNumber);

        // Venezuela: 10-11 digits (with/without country code)
        if ($empresa->pais->codigo_telefonico === '+58') {
            return preg_match('/^(58)?[24]\d{9}$/', $cleanNumber) ||
                   preg_match('/^0[24]\d{9}$/', $cleanNumber);
        }

        // Generic validation for other countries
        return strlen($cleanNumber) >= 8 && strlen($cleanNumber) <= 15;
    }

    private function updateCartCount()
    {
        if (\Auth::check()) {
            $this->cartCount = \Cart::session(userID())->getTotalQuantity();
        } else {
            $this->cartCount = 0;
        }
    }

    private function getCategoriasWithCount()
    {
        return Cache::remember('active_categories_with_product_count', 600, function() {
            return \App\Models\Categoria::where('activo', 1)
                ->withCount(['productos' => function($query) {
                    $query->where('status', 1);
                }])
                ->having('productos_count', '>', 0)
                ->get();
        });
    }

    private function getMarcasWithCount()
    {
        return Cache::remember('active_brands_with_product_count', 600, function() {
            return \App\Models\Marca::where('activo', 1)
                ->withCount(['productos' => function($query) {
                    $query->where('status', 1);
                }])
                ->having('productos_count', '>', 0)
                ->get();
        });
    }

    private function format_money($amount)
    {
        if ($this->currency === 'usd') {
            return '$' . number_format($amount, 2, '.', ',');
        } else {
            return 'Bs. ' . number_format($amount, 2, ',', '.');
        }
    }
}
