<?php

namespace App\Livewire\Front\ContactMessage;

use App\Models\ContactMessage;
use App\Models\Empresa;
use Livewire\Component;
use Livewire\WithFileUploads;
use App\Services\WhatsAppService;

class Index extends Component
{
    use WithFileUploads;

    public $nombre = '';
    public $apellido = '';
    public $telefono = '';
    public $email = '';
    public $mensaje = '';
    public $imagen;

    public $empresa;
    public $mensajeEnviado = false;
    public $mostrarFormulario = true;

    protected $rules = [
        'nombre' => 'required|string|max:50',
        'apellido' => 'required|string|max:50',
        'telefono' => 'required|string|max:20',
        'email' => 'required|email|max:100',
        'mensaje' => 'required|string|max:250',
        'imagen' => 'nullable|image|max:2048', // 2MB max
    ];

    protected $messages = [
        'nombre.required' => 'El nombre es requerido.',
        'nombre.max' => 'El nombre no puede exceder 50 caracteres.',
        'apellido.required' => 'El apellido es requerido.',
        'apellido.max' => 'El apellido no puede exceder 50 caracteres.',
        'telefono.required' => 'El teléfono es requerido.',
        'telefono.max' => 'El teléfono no puede exceder 20 caracteres.',
        'email.required' => 'El email es requerido.',
        'email.email' => 'Debe ingresar un email válido.',
        'email.max' => 'El email no puede exceder 100 caracteres.',
        'mensaje.required' => 'El mensaje es requerido.',
        'mensaje.max' => 'El mensaje no puede exceder 250 caracteres.',
        'imagen.image' => 'La imagen debe ser un archivo de imagen.',
        'imagen.max' => 'La imagen no puede pesar más de 2MB.',
    ];

    public function mount()
    {
        // Obtener la empresa actual (puedes ajustar esto según tu lógica)
        $this->empresa = Empresa::first() ?? Empresa::find(1);
    }

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    public function enviarMensaje()
    {
        $this->validate();

        try {
            $rutaImagen = null;

            if ($this->imagen) {
                $rutaImagen = $this->imagen->store('contact-messages', 'public');
            }

            $contactMessage = ContactMessage::create([
                'nombre' => $this->nombre,
                'apellido' => $this->apellido,
                'telefono' => $this->telefono,
                'email' => $this->email,
                'mensaje' => $this->mensaje,
                'empresa_id' => $this->empresa?->id ?? 1,
            ]);

            // Enviar notificación por WhatsApp
            $this->enviarWhatsApp($contactMessage);

            // Limpiar el formulario
            $this->reset(['nombre', 'apellido', 'telefono', 'email', 'mensaje', 'imagen']);

            $this->mensajeEnviado = true;
            $this->mostrarFormulario = false;

            session()->flash('success', '¡Mensaje enviado exitosamente! Nos pondremos en contacto contigo pronto.');

        } catch (\Exception $e) {
            dd($e);
            session()->flash('error', 'Error al enviar el mensaje. Por favor, inténtalo de nuevo.');
        }
    }

    private function enviarWhatsApp($contactMessage)
    {
        try {
            $mensajeWhatsApp = "*Nuevo mensaje de contacto*\n\n";
            $mensajeWhatsApp .= "*De:* {$contactMessage->nombre_completo}\n";
            $mensajeWhatsApp .= "*Teléfono:* {$contactMessage->telefono}\n";
            $mensajeWhatsApp .= "*Email:* {$contactMessage->email}\n\n";
            $mensajeWhatsApp .= "*Mensaje:*\n{$contactMessage->mensaje}\n\n";
            $mensajeWhatsApp .= "_Recibido el {$contactMessage->created_at->format('d/m/Y H:i')}_";

            // Obtener el número de WhatsApp del administrador o de la empresa
            $telefonoAdmin = $this->empresa?->telefono ?? '584121234567'; // Número por defecto

            // Aquí deberías integrar con tu servicio de WhatsApp
            // Por ahora, marcamos como enviado
            $contactMessage->markAsWhatsAppSent();

            // Si tienes un servicio de WhatsApp, descomenta y ajusta:
            $whatsappService = new \App\Services\WhatsAppService();
            $telefonoFormateado = $whatsappService->cleanPhoneNumber($telefonoAdmin);
             $result = $whatsappService->sendMessage($telefonoFormateado, $message);
            if ($result) {
                $contactMessage->markAsWhatsAppSent();
            } else {
                $contactMessage->markAsWhatsAppFailed();
            }
        } catch (\Exception $e) {
            // Log del error pero no interrumpimos el flujo principal
            \Log::error('Error al enviar WhatsApp: ' . $e->getMessage());
        }
    }


    public function mostrarNuevoFormulario()
    {
        $this->reset(['nombre', 'apellido', 'telefono', 'email', 'mensaje', 'imagen', 'mensajeEnviado', 'mostrarFormulario']);
        $this->mostrarFormulario = true;
    }

    public function render()
    {
        return view('livewire.front.contact-message.index');
    }
}
