<?php

namespace App\Livewire\Front\Checkout;

use App\Models\Cart;
use App\Models\DetallePedido;
use App\Models\HistoricoPedido;
use App\Models\Pedido;
use App\Models\Producto;
use App\Models\Serie;
use App\Models\Tasa;
use App\Models\Banco;
use App\Models\ReferenciaBancaria;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use App\Services\WhatsAppService;

class Index extends Component
{

    public $name, $telefono, $address, $metodoPago, $note = '';
    public $cartItems, $cartTotal, $cartCount;
    public $step = 1; // 1: Info, 2: Pago, 3: Confirmación

    // Nuevas propiedades para el sistema de pagos flexibles
    public $paymentMethods = []; // Array de métodos de pago
    public $selectedPaymentMethod = '';
    public $paymentAmount = 0;
    public $distributedAmount = 0;
    public $remainingAmount = 0;

    // Datos de referencia bancaria (movidos a paymentMethods)
    //public $bancos = [];

    // Propiedades para geolocalización (compatibilidad con JS existente)
    public $city, $country, $county, $municipality, $road, $postcode, $state, $office;
    public $latitud, $longitud;
    public $locationAccuracy; // Nueva propiedad para almacenar precisión
    public $currency; // Nueva propiedad para almacenar precisión

    protected $listeners = [];



    protected $rules = [
        'name' => 'required|string|min:3|max:255',
        'telefono' => 'required|string|min:10|max:20',
        'address' => 'required|string|min:10|max:500',
        'note' => 'nullable|string|max:500',
        'paymentMethods' => 'required|array|min:1',
        'paymentMethods.*.method' => 'required|string|in:Efectivo,Efectivo Bs,Transferencia,Pago Móvil',
        'paymentMethods.*.amount' => 'required|numeric|min:0.01',
        'paymentMethods.*.banco_id' => 'required_if:paymentMethods.*.method,Transferencia,Pago Móvil',
        'paymentMethods.*.referencia' => 'required_if:paymentMethods.*.method,Transferencia,Pago Móvil|string|max:50',
        'paymentMethods.*.documento' => 'required_if:paymentMethods.*.method,Pago Móvil|string|max:20',
        'paymentMethods.*.numero_telefono' => 'required_if:paymentMethods.*.method,Pago Móvil|string|max:20',
        'paymentMethods.*.numero_cuenta' => 'required_if:paymentMethods.*.method,Transferencia|string|max:30',
        'paymentMethods.*.tipo_cuenta' => 'required_if:paymentMethods.*.method,Transferencia|in:ahorro,corriente',
    ];

    protected $messages = [
        'name.required' => 'El nombre es obligatorio',
        'name.min' => 'El nombre debe tener al menos 3 caracteres',
        'telefono.required' => 'El teléfono es obligatorio',
        'telefono.min' => 'El teléfono debe tener al menos 10 caracteres',
        'address.required' => 'La dirección es obligatoria',
        'address.min' => 'La dirección debe tener al menos 10 caracteres',
        'note.max' => 'La nota no puede exceder 500 caracteres',
        'paymentMethods.required' => 'Debe seleccionar al menos un método de pago',
        'paymentMethods.min' => 'Debe seleccionar al menos un método de pago',
        'paymentMethods.*.method.required' => 'El método de pago es obligatorio',
        'paymentMethods.*.amount.required' => 'El monto es obligatorio',
        'paymentMethods.*.amount.min' => 'El monto debe ser mayor a 0',
        'paymentMethods.*.banco_id.required_if' => 'Debe seleccionar un banco',
        'paymentMethods.*.referencia.required_if' => 'El número de referencia es obligatorio',
        'paymentMethods.*.documento.required_if' => 'El documento es obligatorio para pagos móviles',
        'paymentMethods.*.numero_telefono.required_if' => 'El número de teléfono es obligatorio para pagos móviles',
        'paymentMethods.*.numero_cuenta.required_if' => 'El número de cuenta es obligatorio para transferencias',
        'paymentMethods.*.tipo_cuenta.required_if' => 'El tipo de cuenta es obligatorio para transferencias',
    ];


    public function mount()
    {
        \Log::info('🎯 Checkout component montado', [
            'user_id' => auth()->id(),
            'step' => $this->step
        ]);

        // Verificar que hay productos en el carrito
        $this->loadCartData();

        if ($this->cartCount == 0) {
            return redirect('/front/cart');
        }

        $this->name = Auth::user()->name;
        $this->telefono = Auth::user()->telefono;



        $this->currency = (session()->has('currency')) ? session()->get('currency') : 'usd';


        $this->remainingAmount = $this->cartTotal;
    }

    public function loadCartData()
    {
        $this->cartItems = \Cart::session(userID())->getContent();
        $this->cartTotal = \Cart::session(userID())->getTotal();
        $this->cartCount = \Cart::session(userID())->getTotalQuantity();
        $this->remainingAmount = $this->cartTotal;
    }

    public function render()
    {

        // Calcular montos distribuidos
        $this->calculateDistributedAmount();
        $bancos = Banco::activos()->orderBy('nombre')->get();
        $wishlists = collect(); // Variable vacía para evitar error en modals
        return view('livewire.front.checkout.index',compact('bancos', 'wishlists'))
            ->layout('layouts.front.app');
    }

    public function nextStep()
    {
        try {
            if ($this->step == 1) {
                $this->validate([
                    'name' => 'required|string|min:3|max:255',
                    'telefono' => 'required|string|min:10|max:20',
                    'address' => 'required|string|min:10|max:500',
                ]);

                // Validar que se haya obtenido la ubicación
                if (empty($this->latitud) || empty($this->longitud)) {
                    $this->dispatch('alert', [
                        'title' => 'Ubicación requerida',
                        'text' => 'Por favor, haz clic en el botón de ubicación para obtener tu dirección automáticamente',
                        'icon' => 'warning',
                    ]);
                    return;
                }
            } elseif ($this->step == 2) {
                // Validar que se haya distribuido el monto completo
                if (abs($this->distributedAmount - $this->cartTotal) > 0.01) { // Tolerancia para errores de punto flotante
                    $this->dispatch('alert', [
                        'title' => 'Monto incorrecto',
                        'text' => 'El monto distribuido (' . format_money($this->distributedAmount) . ') debe ser igual al total del pedido (' . format_money($this->cartTotal) . ')',
                        'icon' => 'error',
                    ]);
                    return;
                }

                // Validar los métodos de pago
                $this->validate([
                    'paymentMethods' => 'required|array|min:1',
                    'paymentMethods.*.method' => 'required|string|in:Efectivo,Efectivo Bs,Transferencia,Pago Móvil',
                    'paymentMethods.*.amount' => 'required|numeric|min:0.01',
                    'paymentMethods.*.banco_id' => 'exclude_if:paymentMethods.*.method,Efectivo,Efectivo Bs|required|integer|exists:bancos,id',
                    'paymentMethods.*.referencia' => 'exclude_if:paymentMethods.*.method,Efectivo,Efectivo Bs|required|string|max:50',
                    'paymentMethods.*.documento' => 'exclude_if:paymentMethods.*.method,Efectivo,Efectivo Bs,Transferencia|required|string|max:20',
                    'paymentMethods.*.numero_telefono' => 'exclude_if:paymentMethods.*.method,Efectivo,Efectivo Bs,Transferencia|required|string|max:20',
                    'paymentMethods.*.numero_cuenta' => 'exclude_if:paymentMethods.*.method,Efectivo,Efectivo Bs,Pago Móvil|required|string|max:30',
                    'paymentMethods.*.tipo_cuenta' => 'exclude_if:paymentMethods.*.method,Efectivo,Efectivo Bs,Pago Móvil|required|in:ahorro,corriente',
                ]);
            }

            $this->step++;
        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error de validación',
                'text' => 'Por favor, verifica que todos los campos requeridos estén correctamente llenos. Detalle: ' . $e->getMessage(),
                'icon' => 'error',
            ]);
        }
    }

    public function previousStep()
    {
        $this->step--;
    }

    public function updatedSelectedPaymentMethod()
    {
        // Resetear el monto cuando cambia el método de pago
        $this->paymentAmount = 0;
    }

    public function addPaymentMethod()
    {
        // Validar que se haya seleccionado un método y un monto
        if (!$this->selectedPaymentMethod || !$this->paymentAmount || $this->paymentAmount <= 0) {
            $this->dispatch('alert', [
                'title' => 'Datos incompletos',
                'text' => 'Por favor, selecciona un método de pago e ingresa un monto válido',
                'icon' => 'error',
            ]);
            return;
        }

        // Obtener la tasa de cambio del día
        if(tasa() && empresa() == 'Venezuela')
        {
            $tasaModel = tasa();
            if (!$tasaModel) {
                $this->dispatch('alert', [
                    'title' => 'Error',
                    'text' => 'No se pudo obtener la tasa de cambio del día',
                    'icon' => 'error',
                ]);
                return;
            }
            $tasa = $tasaModel->usd_rate;

            // Calcular el monto en Bs si es necesario
            $amountInUSD = $this->paymentAmount;
            $amountInBs = $amountInUSD * $tasa;
        }
        else
        {
              // Calcular el monto en Bs si es necesario
              $amountInUSD = $this->paymentAmount;
              $amountInBs = 0;
        }

        // Validar que el monto no exceda el restante (con tolerancia para errores de punto flotante)
        if ($amountInUSD > $this->remainingAmount + 0.01) {
            $this->dispatch('alert', [
                'title' => 'Monto excedido',
                'text' => 'El monto ingresado (' . format_money($amountInUSD) . ') excede el monto restante por pagar (' . format_money($this->remainingAmount) . ')',
                'icon' => 'error',
            ]);
            return;
        }

        // Crear el nuevo método de pago con campos base
        $newPayment = [
            'method' => $this->selectedPaymentMethod,
            'amount' => $amountInUSD,
            'amount_bs' => $amountInBs,
        ];

        // Agregar campos específicos según el método de pago
        if (in_array($this->selectedPaymentMethod, ['Transferencia', 'Pago Móvil'])) {
            $newPayment['banco_id'] = null;
            $newPayment['referencia'] = null;
        }

        if ($this->selectedPaymentMethod == 'Pago Móvil') {
            $newPayment['documento'] = null;
            $newPayment['numero_telefono'] = null;
        }

        if ($this->selectedPaymentMethod == 'Transferencia') {
            $newPayment['numero_cuenta'] = null;
            $newPayment['tipo_cuenta'] = null;
        }

        $this->paymentMethods[] = $newPayment;

        // Resetear campos
        $this->selectedPaymentMethod = '';
        $this->paymentAmount = 0;

        // Recalcular montos
        $this->calculateDistributedAmount();
    }

    public function removePaymentMethod($index)
    {
        unset($this->paymentMethods[$index]);
        $this->paymentMethods = array_values($this->paymentMethods); // Reindexar array
        $this->calculateDistributedAmount();
    }

    public function calculateDistributedAmount()
    {
        $this->distributedAmount = array_sum(array_column($this->paymentMethods, 'amount'));
        $this->remainingAmount = $this->cartTotal - $this->distributedAmount;
    }

    public function getTotalByPaymentMethod()
    {
        return $this->cartTotal;
    }

    public function removeItem($id)
    {
        try {
            \Cart::session(userID())->remove($id);
            $this->loadCartData();

            if ($this->cartCount == 0) {
                return redirect('/front/cart');
            }

            $this->dispatch('alert', [
                'title' => '¡Eliminado!',
                'text' => 'Producto eliminado del carrito',
                'icon' => 'success',
            ]);
        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo eliminar el producto del carrito',
                'icon' => 'error',
            ]);
        }
    }

    public function processOrder()
    {


        // Validación adicional antes de procesar
        if ($this->cartCount == 0) {
            $this->dispatch('alert', [
                'title' => 'Carrito vacío',
                'text' => 'No hay productos en el carrito',
                'icon' => 'error',
            ]);
            return redirect('/front/cart');
        }

        if (abs($this->distributedAmount - $this->cartTotal) > 0.01) {
            $this->dispatch('alert', [
                'title' => 'Monto incorrecto',
                'text' => 'El monto distribuido (' . format_money($this->distributedAmount) . ') debe ser igual al total del pedido (' . format_money($this->cartTotal) . ')',
                'icon' => 'error',
                ]);
                return;
                }

                if (count($this->paymentMethods) == 0) {
                    $this->dispatch('alert', [
                'title' => 'Método de pago requerido',
                'text' => 'Debe seleccionar al menos un método de pago',
                'icon' => 'error',
                ]);
                return;
                }

                // Realizar validación general antes de cualquier operación

                try {
                    DB::beginTransaction();

            // Log de inicio del proceso
            \Log::info('Iniciando proceso de orden', [
                'user_id' => Auth::id(),
                'cart_total' => $this->cartTotal,
                'payment_methods_count' => count($this->paymentMethods),
                'step' => $this->step
                ]);

                if(tasa() && empresa() == 'Venezuela')
                    {
                        // Obtener tasa del día
                $tasaModel = tasa();
                if (!$tasaModel) {
                    throw new \Exception('No se pudo obtener la tasa de cambio del día');
                    }
                    $tasa = $tasaModel->usd_rate;
                    }
                    else
                        {
                 $tasa = 1;
                 }


                 // Generar correlativo del pedido usando el modelo Serie correctamente
                 $ultima_serie = Serie::where('empresa_id', Auth::user()->empresa_id)->first();

            if (!$ultima_serie) {
                throw new \Exception('No se encontró la configuración de series');
                }

                // Incrementar el número de serie para el código del pedido
                $nroPedido = (int) $ultima_serie->serie + 1;
                $nuevo_correlativo              = $nroPedido;
                $ultima_serie->serie = $nuevo_correlativo;
                $ultima_serie->save();

                // Determinar el método de pago principal
                $metodoPagoPrincipal = count($this->paymentMethods) == 1 ? $this->paymentMethods[0]['method'] : 'Mixto';

                // Crear el pedido
                $pedido = new Pedido();
                $pedido->fecha = now();
                if ($this->currency == 'usd') {
                    $pedido->total_usd = $this->cartTotal ;
                    $pedido->total_bs = $this->cartTotal * $tasa;
                    $pedido->moneda    = $this->currency;
                    } else {
                        $pedido->total_bs = $this->cartTotal /  $tasa;
                        $pedido->total_usd = $this->cartTotal ;
                        $pedido->moneda    = $this->currency;
                    }

                    $pedido->codigo = str_pad($nroPedido, 4, '0', STR_PAD_LEFT);
                    $pedido->metodo_pago = $metodoPagoPrincipal;
                    $pedido->estado = 'Pendiente';
                    $pedido->ubicacion = $this->address;
                    $pedido->longitud = $this->longitud ?? '0';
                    $pedido->latitud  = $this->latitud ?? '0';

            $pedido->empresaId = Auth::user()->empresa_id;
            $pedido->userId = Auth::user()->id;
            $pedido->save();

            // Guardar detalles de pago por cada método
            foreach ($this->paymentMethods as $payment) {
                $pedidoPago = new \App\Models\PedidoPago();
                $pedidoPago->pedidoId = $pedido->id;
                $pedidoPago->referencia = $payment['referencia'] ?? null;
                $pedidoPago->metodo_pago = $payment['method'];
                 if ($this->currency == 'usd') {
                     $pedidoPago->total_usd = $payment['amount'];
                     $pedidoPago->total_bs = $payment['amount'] * $tasa;
                     } else {
                         $pedidoPago->total_usd = $payment['amount'] / $tasa;
                $pedidoPago->total_bs = $payment['amount'];
                }
                $pedidoPago->nota = $this->note ?? 'N/A';
                $pedidoPago->empresaId = Auth::user()->empresa_id;
                $pedidoPago->userId = Auth::user()->id;
                $pedidoPago->save();
                }

                // Guardar detalles de los productos
                foreach ($this->cartItems as $item) {

                    $producto = Producto::find($item->associatedModel->id);
                if (!$producto) {
                    throw new \Exception('Producto no encontrado: ' . $item->name);
                    }

                    $detalle = new DetallePedido();
                $detalle->fecha = now();
                $detalle->total_usd = $item->price * $item->quantity;
                $detalle->total_bs = $detalle->total_usd * $tasa;
                $detalle->quantity = $item->quantity;
                $detalle->price = $item->price;
                $detalle->empresaId = Auth::user()->empresa_id;
                $detalle->userId = Auth::user()->id;
                $detalle->productoId = $producto->id;
                $detalle->pedidoId = $pedido->id;

                 if (isset($item->attributes->variant_id) || $item->attributes->variant_id) {
                    $detalle->variant_id = $item->attributes->variant_id;
                    $detalle->product_name = $item->name; // Nombre completo con variante
                } else {
                    $detalle->product_name = $producto->name; // Nombre del producto principal
                }



                $detalle->save();
            }

            // Registrar historial
            $historico = new HistoricoPedido();
            $historico->fecha = now();
            $historico->titulo = 'Orden creada';
            $historico->descripcion = "El pedido #: {$pedido->codigo} Se ha realizado.";
            $historico->empresaId = Auth::user()->empresa_id;
            $historico->userId = Auth::user()->id;
            $historico->pedidoId = $pedido->id;
            $historico->save();

            DB::commit();

            // Enviar notificación por WhatsApp usando Jobs
            try {
                // Cargar relaciones necesarias para WhatsApp
                $pedido->load(['user.empresa.pais', 'detalles.producto']);
                \App\Jobs\SendOrderWhatsAppNotification::dispatch($pedido);

                \Artisan::call('queue:work', ['--once' => true, '--quiet' => true]);

                \Log::info('Notificación WhatsApp enviada para pedido', [
                    'pedido_id' => $pedido->id,
                    'codigo' => $pedido->codigo
                ]);
            } catch (\Exception $e) {
                \Log::error('Error al enviar notificación WhatsApp', [
                    'pedido_id' => $pedido->id,
                    'error' => $e->getMessage()
                ]);
            }

            // Limpiar carrito y redirigir
            \Cart::session(userID())->clear();

            $this->dispatch('alert', [
                'title' => '¡Pedido procesado!',
                'text' => '¡Pedido generado satisfactoriamente!',
                'icon' => 'success',
            ]);

            return redirect('front/pedido/detalle/' . $pedido->codigo);
        } catch (\Throwable $th) {

            DB::rollBack();
            \Log::error('Error procesando pedido: ' . $th->getMessage(), [
                'user_id' => Auth::id(),
                'error' => $th->getMessage(),
                'file' => $th->getFile(),
                'line' => $th->getLine(),
                'trace' => $th->getTraceAsString()
            ]);

            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'Ocurrió un error al procesar el pedido: ' . $th->getMessage() . '. Por favor, inténtalo nuevamente.',
                'icon' => 'error',
            ]);
        }
    }




    function random_digits($length)
    {
        $result = '';

        for ($i = 0; $i < $length; $i++) {
            $result .= random_int(0, 9);
        }

        return $result;
    }

    public function getCurrentLocation()
    {
        $this->dispatch('getLocation');
    }

    public function updateLocation($address, $latitude, $longitude, $accuracy = null)
    {
        $this->address = $address;
        $this->latitud = $latitude;
        $this->longitud = $longitude;
        $this->locationAccuracy = $accuracy;

        $message = 'Tu dirección actual ha sido detectada';
        if ($accuracy) {
            $message .= ". Precisión: {$accuracy} metros";
        }

        $this->dispatch('alert', [
            'title' => '¡Ubicación obtenida!',
            'text' => $message,
            'icon' => 'success',
        ]);
    }

    public function updatedAddress()
    {
        // Este método se ejecuta cuando se actualiza la dirección
        // Las coordenadas se guardarán cuando se procese el pedido
    }

    // Método de prueba para verificar que el botón funciona
    public function testButton()
    {
        $this->dispatch('alert', [
            'title' => '¡Botón funcionando!',
            'text' => 'El botón de prueba ha sido clickeado exitosamente',
            'icon' => 'success',
        ]);
    }
}
