<?php

namespace App\Livewire\Admin\Ventas;

use App\Models\Venta;
use App\Models\Cliente;
use App\Models\Caja;
use App\Models\DetalleVenta;
use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;

class Index extends Component
{
    use WithPagination, HasDynamicLayout, HasRegionalFormatting;

    // Filtros
    public $search = '';
    public $fechaInicio = '';
    public $fechaFin = '';
    public $estado = '';
    public $metodoPago = '';
    public $tipoDocumento = '';
    public $clienteId = '';
    public $cajaId = '';

    // Ordenamiento
    public $orderBy = 'created_at';
    public $orderDirection = 'desc';

    // Configuración regional de moneda
    protected $currencySymbol = '$';
    protected $currencyFormat = '#,##0.00';
    protected $dateFormat = 'd/m/Y';

    // Estados disponibles
    public $estados = [
        'pendiente' => 'Pendiente',
        'pagado' => 'Pagado',
        'cancelado' => 'Anulado',
    ];

    // Métodos de pago
    public $metodosPago = [
        'efectivo' => 'Efectivo',
        'transferencia' => 'Transferencia',
        'tarjeta' => 'Tarjeta',
        'pago_movil' => 'Pago Móvil',
        'punto_de_venta' => 'Punto de Venta',
    ];

    // Tipos de documento
    public $tiposDocumento = [
        'factura' => 'Factura',
        'boleta' => 'Boleta',
        'nota_credito' => 'Nota de Crédito',
        'recibo' => 'Recibo',
    ];

    protected $queryString = [
        'search' => ['except' => ''],
        'fechaInicio' => ['except' => ''],
        'fechaFin' => ['except' => ''],
        'estado' => ['except' => ''],
        'metodoPago' => ['except' => ''],
        'tipoDocumento' => ['except' => ''],
        'clienteId' => ['except' => ''],
        'cajaId' => ['except' => ''],
    ];

    public function mount()
    {
        // Inicializar configuración regional
        $this->bootHasRegionalFormatting();

        // Obtener configuración regional
        $regionalConfig = $this->getRegionalConfig();

        // Establecer símbolo y formato de moneda
        $this->currencySymbol = $regionalConfig['currency_symbol'] ?? '$';
        $this->currencyFormat = $regionalConfig['currency_format'] ?? '#,##0.00';
        $this->dateFormat = $regionalConfig['date_format'] ?? 'd/m/Y';

        // Establecer fechas por defecto (mes actual)
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');

        // Calcular estadísticas iniciales
        $this->calcularEstadisticas();
    }

    public function calcularEstadisticas()
    {
        // Este método se puede usar para precalcular estadísticas si es necesario
        // Por ahora las estadísticas se calculan mediante propiedades computadas
    }

    public function getTotalVentasProperty()
    {
        return Venta::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', '!=', 'cancelado')
            ->whereDate('created_at', '>=', $this->fechaInicio)
            ->whereDate('created_at', '<=', $this->fechaFin)
            ->count();
    }

    public function getTotalMontoProperty()
    {
        return Venta::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', '!=', 'cancelado')
            ->whereDate('created_at', '>=', $this->fechaInicio)
            ->whereDate('created_at', '<=', $this->fechaFin)
            ->sum('total');
    }

    public function getPromedioVentaProperty()
    {
        $totalVentas = $this->totalVentas;
        $totalMonto = Venta::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', '!=', 'cancelado')
            ->whereDate('created_at', '>=', $this->fechaInicio)
            ->whereDate('created_at', '<=', $this->fechaFin)
            ->sum('total');
        
        return $totalVentas > 0 ? $totalMonto / $totalVentas : 0;
    }

    public function getVentasHoyProperty()
    {
        return Venta::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', '!=', 'cancelado')
            ->whereDate('created_at', today())
            ->count();
    }

    public function getMontoHoyProperty()
    {
        return Venta::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', '!=', 'cancelado')
            ->whereDate('created_at', today())
            ->sum('total');
    }

    public function getVentasMesProperty()
    {
        return Venta::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', '!=', 'cancelado')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();
    }

    public function getMontoMesProperty()
    {
        $monto = Venta::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', '!=', 'cancelado')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('total');
        return $this->formatMoney($monto);
    }

    public function getVentasPorEstadoProperty()
    {
        $estados = ['pendiente', 'pagado', 'cancelado'];
        $result = [];

        foreach ($estados as $estado) {
            $ventas = Venta::where('empresa_id', auth()->user()->empresa_id)
                ->where('sucursal_id', auth()->user()->sucursal_id)
                ->where('estado', $estado)
                ->whereDate('created_at', '>=', $this->fechaInicio)
                ->whereDate('created_at', '<=', $this->fechaFin);

            $nombre = $estado === 'cancelado' ? 'Anulado' : ucfirst($estado);
            $result[] = [
                'nombre' => $nombre,
                'cantidad' => $ventas->count(),
                'monto' => $this->formatMoney($ventas->sum('total'))
            ];
        }

        return $result;
    }

    public function getVentasPorMetodoPagoProperty()
    {
        $metodos = ['efectivo', 'transferencia', 'tarjeta', 'pago_movil', 'punto_de_venta'];
        $result = [];

        foreach ($metodos as $metodo) {
            $ventas = Venta::where('empresa_id', auth()->user()->empresa_id)
                ->where('sucursal_id', auth()->user()->sucursal_id)
                ->where('metodo_pago', $metodo)
                ->whereDate('created_at', '>=', $this->fechaInicio)
                ->whereDate('created_at', '<=', $this->fechaFin);

            $result[] = [
                'nombre' => ucfirst(str_replace('_', ' ', $metodo)),
                'cantidad' => $ventas->count(),
                'monto' => $this->formatMoney($ventas->sum('total'))
            ];
        }

        return $result;
    }

    public function render()
    {
        $query = Venta::with(['cliente', 'usuario', 'caja', 'serie'])
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id);

        // Aplicar búsqueda
        if ($this->search) {
            $query->where(function($q) {
                $q->where('codigo', 'like', '%' . $this->search . '%')
                  ->orWhere('numero_documento', 'like', '%' . $this->search . '%');
            });
        }

        // Aplicar filtros de fecha
        if ($this->fechaInicio) {
            $query->whereDate('created_at', '>=', $this->fechaInicio);
        }
        if ($this->fechaFin) {
            $query->whereDate('created_at', '<=', $this->fechaFin);
        }

        // Aplicar filtros
        if ($this->estado) {
            $query->where('estado', $this->estado);
        }
        if ($this->metodoPago) {
            $query->where('metodo_pago', $this->metodoPago);
        }
        if ($this->tipoDocumento) {
            $query->where('tipo_documento', $this->tipoDocumento);
        }
        if ($this->clienteId) {
            $query->where('cliente_id', $this->clienteId);
        }
        if ($this->cajaId) {
            $query->where('caja_id', $this->cajaId);
        }

        // Aplicar ordenamiento
        $query->orderBy($this->orderBy, $this->orderDirection);

        $ventas = $query->paginate(15);
        $clientes = Cliente::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->get();
        $cajas = Caja::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->get();

        return view('livewire.admin.ventas.index', [
            'ventas' => $ventas,
            'clientes' => $clientes,
            'cajas' => $cajas,
        ])->layout($this->getLayout());
    }

    public function orderBy($field)
    {
        if ($this->orderBy === $field) {
            $this->orderDirection = $this->orderDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->orderBy = $field;
            $this->orderDirection = 'asc';
        }
    }

    public function limpiarFiltros()
    {
        $this->search = '';
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');
        $this->estado = '';
        $this->metodoPago = '';
        $this->tipoDocumento = '';
        $this->clienteId = '';
        $this->cajaId = '';
        $this->clienteId = '';
        $this->cajaId = '';
        $this->resetPage();
    }

    public function confirmarAnulacion($ventaId)
    {
        $venta = Venta::find($ventaId);
        if (!$venta) {
            session()->flash('error', 'Venta no encontrada');
            return;
        }

        if ($venta->estado === 'cancelado') {
            session()->flash('error', 'La venta ya está anulada');
            return;
        }

        $venta->estado = 'cancelado';
        $venta->save();

        // Restaurar stock de los productos
        $detalles = DetalleVenta::where('venta_id', $ventaId)->get();
        foreach ($detalles as $detalle) {
            if ($detalle->producto) {
                $detalle->producto->quantity += $detalle->cantidad;
                $detalle->producto->save();
            }
        }

        session()->flash('message', 'Venta anulada exitosamente');
        $this->render();
    }
}
