<?php

namespace App\Livewire\Admin\Users\Profile;

use App\Traits\HasDynamicLayout;
use Livewire\Component;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class ProfileStats extends Component
{
    use HasDynamicLayout;

    public $user;
    public $period = 'week';

    public function mount(User $user)
    {
        $this->user = $user;
    }

    public function setPeriod($period)
    {
        $this->period = $period;
        $this->dispatch('updateChart', chartData: $this->getChartData());
    }

    public function getDetailedStats()
    {
        $userId = $this->user->id;
        $now = Carbon::now();

        switch ($this->period) {
            case 'week':
                $startDate = $now->copy()->subDays(6)->startOfDay();
                $endDate = $now->copy()->endOfDay();
                break;
            case 'month':
                $startDate = $now->copy()->subDays(29)->startOfDay();
                $endDate = $now->copy()->endOfDay();
                break;
            case 'quarter':
                $startDate = $now->copy()->subMonths(2)->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
                break;
            case 'year':
                $startDate = $now->copy()->subMonths(11)->startOfMonth();
                $endDate = $now->copy()->endOfMonth();
                break;
            default:
                $startDate = $now->copy()->subDays(29)->startOfDay();
                $endDate = $now->copy()->endOfDay();
                break;
        }

        $currentPeriodStats = $this->getPeriodStats($userId, $startDate, $endDate);
        $previousStart = $startDate->copy()->sub($this->getPeriodInterval());
        $previousEnd = $endDate->copy()->sub($this->getPeriodInterval());
        $previousPeriodStats = $this->getPeriodStats($userId, $previousStart, $previousEnd);
        $changes = $this->calculateChanges($currentPeriodStats, $previousPeriodStats);

        return [
            'current' => $currentPeriodStats,
            'previous' => $previousPeriodStats,
            'changes' => $changes,
            'period_name' => $this->getPeriodName()
        ];
    }

    private function getPeriodStats($userId, $start, $end)
    {
        $user = \App\Models\User::find($userId);
        $empleado = \App\Models\Empleado::where('user_id', $userId)->first();

        if (!$empleado) {
            if (!$user->empresa_id) return ['total_pedidos' => 0, 'pedidos_completados' => 0, 'pedidos_pendientes' => 0, 'pedidos_cancelados' => 0, 'total_gastado' => 0, 'promedio_pedido' => 0];

            $pedidos = \App\Models\Pedido::where('empresaId', $user->empresa_id)
                ->whereBetween('created_at', [$start, $end]);
        } else {
            $pedidos = \App\Models\Pedido::whereHas('empleados', function($q) use ($empleado) {
                    $q->where('empleadoId', $empleado->id);
                })
                ->whereBetween('created_at', [$start, $end]);
        }

        return [
            'total_pedidos' => $pedidos->count(),
            'pedidos_completados' => $pedidos->clone()->where('estado', 'entregado')->count(),
            'pedidos_pendientes' => $pedidos->clone()->where('estado', 'pendiente')->count(),
            'pedidos_cancelados' => $pedidos->clone()->where('estado', 'cancelado')->count(),
            'total_gastado' => $pedidos->clone()->where('estado', 'entregado')->sum('total_usd'),
            'promedio_pedido' => $pedidos->clone()->where('estado', 'entregado')->avg('total_usd') ?: 0,
        ];
    }

    private function getPeriodInterval()
    {
        switch ($this->period) {
            case 'week':
                return \DateInterval::createFromDateString('7 days');
            case 'quarter':
                return \DateInterval::createFromDateString('3 months');
            case 'year':
                return \DateInterval::createFromDateString('1 year');
            default:
                return \DateInterval::createFromDateString('1 month');
        }
    }

    private function calculateChanges($current, $previous)
    {
        $changes = [];
        foreach ($current as $key => $value) {
            $previousValue = $previous[$key] ?? 0;
            if ($previousValue > 0) {
                $changes[$key] = round((($value - $previousValue) / $previousValue) * 100, 1);
            } else {
                $changes[$key] = $value > 0 ? 100 : 0;
            }
        }
        return $changes;
    }

    private function getPeriodName()
    {
        switch ($this->period) {
            case 'week':
                return 'Últimos 7 días';
            case 'quarter':
                return 'Trimestre';
            case 'year':
                return 'Año';
            default:
                return 'Mes';
        }
    }

    public function getChartData()
    {
        $userId = $this->user->id;
        $empleado = \App\Models\Empleado::where('user_id', $userId)->first();
        $now = Carbon::now();
        $data = [];

        if ($this->period == 'year' || $this->period == 'quarter') {
            // Para año: mostrar 12 meses, para trimestre: 3 meses
            $months = $this->period == 'year' ? 12 : 3;
            
            for ($i = $months - 1; $i >= 0; $i--) {
                $monthStart = $now->copy()->subMonths($i)->startOfMonth();
                $monthEnd = $monthStart->copy()->endOfMonth();
                
                if (!$empleado) {
                    if (!$this->user->empresa_id) {
                        $pedidos = 0;
                    } else {
                        $pedidos = \App\Models\Pedido::where('empresaId', $this->user->empresa_id)
                            ->whereBetween('created_at', [$monthStart, $monthEnd])
                            ->count();
                    }
                } else {
                    $pedidos = \App\Models\Pedido::whereHas('empleados', function($q) use ($empleado) {
                            $q->where('empleadoId', $empleado->id);
                        })
                        ->whereBetween('created_at', [$monthStart, $monthEnd])
                        ->count();
                }

                $data[] = [
                    'date' => $monthStart->format('Y-m-01'),
                    'pedidos' => $pedidos,
                    'label' => $monthStart->format('M Y') // Ene 2024
                ];
            }
        } else {
            // Para otros períodos: mostrar por días
            switch ($this->period) {
                case 'week':
                    $days = 7;
                    break;
                case 'month':
                    $days = 30;
                    break;

                default:
                    $days = 30;
                    break;
            }

            for ($i = $days - 1; $i >= 0; $i--) {
                $date = $now->copy()->subDays($i);
                
                if (!$empleado) {
                    if (!$this->user->empresa_id) {
                        $pedidos = 0;
                    } else {
                        $pedidos = \App\Models\Pedido::where('empresaId', $this->user->empresa_id)
                            ->whereDate('created_at', $date)
                            ->count();
                    }
                } else {
                    $pedidos = \App\Models\Pedido::whereHas('empleados', function($q) use ($empleado) {
                            $q->where('empleadoId', $empleado->id);
                        })
                        ->whereDate('created_at', $date)
                        ->count();
                }

                $data[] = [
                    'date' => $date->format('Y-m-d'),
                    'pedidos' => $pedidos,
                    'label' => $this->getDateLabel($date, $this->period)
                ];
            }
        }
        
        return $data;
    }

    private function getDateLabel($date, $period)
    {
        switch ($period) {
            case 'week':
                return $date->format('D j'); // Lun 15
            case 'month':
                return $date->format('j M'); // 15 Ene
            case 'quarter':
            case 'year':
                return $date->format('j/n'); // 15/1
            default:
                return $date->format('j M');
        }
    }

    public function render()
    {
        return view('livewire.admin.users.profile.profile-stats', [
            'stats' => $this->getDetailedStats(),
            'chartData' => $this->getChartData()
        ]);
    }
}
