<?php

namespace App\Livewire\Admin\Users\Profile;

use App\Traits\HasDynamicLayout;
use Livewire\Component;
use App\Models\User;
use App\Models\Pedido;
use Illuminate\Support\Facades\Auth;

class Index extends Component
{
    use HasDynamicLayout;

    public $user;
    public $editMode = false;
    public $name;
    public $email;
    public $phone;
    public $address;
    public $bio;
    public $notifications_email = true;
    public $notifications_sms = false;
    public $notifications_push = true;

    public function mount()
    {
        $this->user = User::with(['empresa', 'sucursal', 'roles'])->findOrFail(auth()->id());
        $this->name = $this->user->name;
        $this->email = $this->user->email;
        $this->phone = $this->user->phone;
        $this->address = $this->user->address;
        $this->bio = $this->user->bio;
        $this->notifications_email = $this->user->notifications_email ?? true;
        $this->notifications_sms = $this->user->notifications_sms ?? false;
        $this->notifications_push = $this->user->notifications_push ?? true;
    }

    public function toggleEditMode()
    {
        $this->editMode = !$this->editMode;
        if (!$this->editMode) {
            $this->mount();
        }
    }

    public function updateProfile()
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $this->user->id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'bio' => 'nullable|string|max:1000',
        ]);

        $this->user->update([
            'name' => $this->name,
            'email' => $this->email,
            'phone' => $this->phone,
            'address' => $this->address,
            'bio' => $this->bio,
            'notifications_email' => $this->notifications_email,
            'notifications_sms' => $this->notifications_sms,
            'notifications_push' => $this->notifications_push,
        ]);

        $this->editMode = false;
        $this->dispatch('alert', [
            'title' => '¡Actualizado!',
            'text' => 'Tu perfil ha sido actualizado correctamente',
            'icon' => 'success'
        ]);
    }

    public function getUserStats()
    {
        $user = Auth::user();
        $empleado = \App\Models\Empleado::where('user_id', $user->id)->first();
        
        // Si no es empleado, mostrar estadísticas generales de la empresa
        if (!$empleado) {
            $empresaId = $user->empresa_id;
            if (!$empresaId) return ['total_pedidos' => 0, 'pedidos_completados' => 0, 'pedidos_pendientes' => 0, 'total_gastado' => 0, 'pedidos_este_mes' => 0, 'tasa_completado' => 0];
            
            $totalPedidos = Pedido::where('empresaId', $empresaId)->count();
            $pedidosCompletados = Pedido::where('empresaId', $empresaId)->where('estado', 'entregado')->count();
            $pedidosPendientes = Pedido::where('empresaId', $empresaId)->where('estado', 'pendiente')->count();
            $totalGastado = Pedido::where('empresaId', $empresaId)->where('estado', 'entregado')->sum('total_usd');
            $pedidosEsteMes = Pedido::where('empresaId', $empresaId)->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year)->count();
        } else {
            // Si es empleado, mostrar sus estadísticas específicas
            $totalPedidos = Pedido::whereHas('empleados', function($q) use ($empleado) { $q->where('empleadoId', $empleado->id); })->count();
            $pedidosCompletados = Pedido::whereHas('empleados', function($q) use ($empleado) { $q->where('empleadoId', $empleado->id); })->where('estado', 'entregado')->count();
            $pedidosPendientes = Pedido::whereHas('empleados', function($q) use ($empleado) { $q->where('empleadoId', $empleado->id); })->where('estado', 'pendiente')->count();
            $totalGastado = Pedido::whereHas('empleados', function($q) use ($empleado) { $q->where('empleadoId', $empleado->id); })->where('estado', 'entregado')->sum('total_usd');
            $pedidosEsteMes = Pedido::whereHas('empleados', function($q) use ($empleado) { $q->where('empleadoId', $empleado->id); })->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year)->count();
        }
            
        return [
            'total_pedidos' => $totalPedidos,
            'pedidos_completados' => $pedidosCompletados,
            'pedidos_pendientes' => $pedidosPendientes,
            'total_gastado' => $totalGastado,
            'pedidos_este_mes' => $pedidosEsteMes,
            'tasa_completado' => $totalPedidos > 0 ? round(($pedidosCompletados / $totalPedidos) * 100, 1) : 0
        ];
    }

    public function getRecentActivity()
    {
        $user = Auth::user();
        $empleado = \App\Models\Empleado::where('user_id', $user->id)->first();
        
        if (!$empleado) {
            // Si no es empleado, mostrar actividad general de la empresa
            $empresaId = $user->empresa_id;
            if (!$empresaId) return collect();
            
            return Pedido::where('empresaId', $empresaId)
                ->with(['detalles.producto'])
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();
        }
        
        return Pedido::whereHas('empleados', function($q) use ($empleado) {
                $q->where('empleadoId', $empleado->id);
            })
            ->with(['detalles.producto'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
    }

    public function render()
    {
        return $this->renderWithLayout('livewire.admin.users.profile.index', [
            'stats' => $this->getUserStats(),
            'recentActivity' => $this->getRecentActivity()
        ], [
            'title' => 'Mi Perfil',
            'description' => 'Gestión de perfil de usuario',
            'breadcrumb' => [
                'admin.dashboard' => 'Dashboard',
                'admin.users.profile' => 'Mi Perfil'
            ]
        ]);
    }
}
