<?php

namespace App\Livewire\Admin\Series;
use App\Traits\HasDynamicLayout;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Serie;
use App\Models\Empresa;
use App\Models\Sucursal;

class Index extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $tipo_documento = '';
    public $estado = '';
    public $empresa_id = '';
    public $sucursal_id = '';
    public $sortBy = 'tipo_documento';
    public $sortDirection = 'asc';
    public $perPage = 15;
    public $showFilters = false;
    
    // Bulk actions
    public $selectedSeries = [];
    public $selectAll = false;

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedSelectAll($value)
    {
        if ($value) {
            $this->selectedSeries = $this->getSeries()->pluck('id')->toArray();
        } else {
            $this->selectedSeries = [];
        }
    }

    public function sortBy($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    public function toggleActivo($id)
    {
        $serie = Serie::findOrFail($id);
        $serie->update(['activo' => !$serie->activo]);

        $this->dispatch('alert', [
            'title' => 'Estado actualizado',
            'text' => 'El estado de la serie se actualizó correctamente',
            'icon' => 'success'
        ]);
    }

    public function bulkToggleActivo()
    {
        if (empty($this->selectedSeries)) {
            $this->dispatch('alert', [
                'title' => 'Sin selección',
                'text' => 'Selecciona al menos una serie',
                'icon' => 'warning'
            ]);
            return;
        }

        Serie::whereIn('id', $this->selectedSeries)->update(['activo' => true]);
        $this->selectedSeries = [];
        $this->selectAll = false;

        $this->dispatch('alert', [
            'title' => 'Series activadas',
            'text' => 'Las series seleccionadas fueron activadas',
            'icon' => 'success'
        ]);
    }

    public function bulkDelete()
    {
        if (empty($this->selectedSeries)) {
            $this->dispatch('alert', [
                'title' => 'Sin selección',
                'text' => 'Selecciona al menos una serie',
                'icon' => 'warning'
            ]);
            return;
        }

        Serie::whereIn('id', $this->selectedSeries)->delete();
        $this->selectedSeries = [];
        $this->selectAll = false;

        $this->dispatch('alert', [
            'title' => 'Series eliminadas',
            'text' => 'Las series seleccionadas fueron eliminadas',
            'icon' => 'success'
        ]);
    }

    public function delete($id)
    {
        Serie::findOrFail($id)->delete();
        
        $this->dispatch('alert', [
            'title' => 'Serie eliminada',
            'text' => 'La serie fue eliminada correctamente',
            'icon' => 'success'
        ]);
    }

    public function clearFilters()
    {
        $this->reset(['search', 'tipo_documento', 'estado', 'empresa_id', 'sucursal_id']);
        $this->resetPage();
    }

    private function getSeries()
    {
        return Serie::query()
            ->when($this->search, fn($q) => $q->where('serie', 'like', "%{$this->search}%"))
            ->when($this->tipo_documento, fn($q) => $q->where('tipo_documento', $this->tipo_documento))
            ->when($this->estado !== '', fn($q) => $q->where('activo', $this->estado))
            ->when($this->empresa_id, fn($q) => $q->where('empresa_id', $this->empresa_id))
            ->when($this->sucursal_id, fn($q) => $q->where('sucursal_id', $this->sucursal_id))
            ->with(['empresa', 'sucursal'])
            ->orderBy($this->sortBy, $this->sortDirection);
    }

    public function render()
    {
        $series = $this->getSeries()->paginate($this->perPage);
        
        $empresas = Empresa::where('status', 1)->get();
        $sucursales = Sucursal::where('status', 1)
            ->when($this->empresa_id, fn($q) => $q->where('empresa_id', $this->empresa_id))
            ->get();

        $stats = [
            'total' => Serie::count(),
            'activas' => Serie::where('activo', true)->count(),
            'inactivas' => Serie::where('activo', false)->count(),
            'por_tipo' => Serie::selectRaw('tipo_documento, count(*) as total')
                ->groupBy('tipo_documento')
                ->pluck('total', 'tipo_documento')
                ->toArray()
        ];

        return $this->renderWithLayout('livewire.admin.series.index', [
            'series' => $series,
            'tipos' => Serie::getTiposDocumento(),
            'empresas' => $empresas,
            'sucursales' => $sucursales,
            'stats' => $stats
        ], [
            'description' => 'Gestión de Series de Documentos',
        ]);
    }
}
