<?php

namespace App\Livewire\Admin\Promociones;

use App\Models\Promotion;
use App\Models\Producto;
use App\Models\Categoria;
use Livewire\Component;
use App\Traits\HasDynamicLayout;

class Edit extends Component
{
    use HasDynamicLayout;

    public Promotion $promotion;

    public $name = '';
    public $description = '';
    public $discount_type = 'percent';
    public $discount_value = '';
    public $buy_qty = 2;
    public $get_qty = 1;
    public $starts_at = '';
    public $ends_at = '';
    public $max_uses = '';
    public $max_uses_per_user = '';
    public $is_active = true;
    public $is_automatic = false;
    public $stackable = false;
    public $priority = 0;
    public $min_amount = '';
    public $selectedProducts = [];
    public $selectedCategories = [];

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'discount_type' => 'required|in:percent,fixed,bxgy',
            'discount_value' => 'required_unless:discount_type,bxgy|nullable|numeric|min:0',
            'buy_qty' => 'required_if:discount_type,bxgy|nullable|integer|min:1',
            'get_qty' => 'required_if:discount_type,bxgy|nullable|integer|min:1',
            'starts_at' => 'nullable|date',
            'ends_at' => 'nullable|date|after_or_equal:starts_at',
            'max_uses' => 'nullable|integer|min:1',
            'max_uses_per_user' => 'nullable|integer|min:1',
            'is_active' => 'boolean',
            'is_automatic' => 'boolean',
            'stackable' => 'boolean',
            'priority' => 'integer|min:0',
            'min_amount' => 'nullable|numeric|min:0',
        ];
    }

    public function mount(Promotion $promotion)
    {
        // Verificar que pertenece a la empresa
        if ($promotion->empresa_id !== auth()->user()->empresa_id) {
            abort(403);
        }

        $this->promotion = $promotion;
        $this->name = $promotion->name;
        $this->description = $promotion->description;
        $this->discount_type = $promotion->discount_type;
        $this->discount_value = $promotion->discount_value;
        $this->buy_qty = $promotion->buy_qty ?? 2;
        $this->get_qty = $promotion->get_qty ?? 1;
        $this->starts_at = $promotion->starts_at?->format('Y-m-d\TH:i');
        $this->ends_at = $promotion->ends_at?->format('Y-m-d\TH:i');
        $this->max_uses = $promotion->max_uses;
        $this->max_uses_per_user = $promotion->max_uses_per_user;
        $this->is_active = $promotion->is_active;
        $this->is_automatic = $promotion->is_automatic;
        $this->stackable = $promotion->stackable;
        $this->priority = $promotion->priority;

        // Cargar productos y categorías seleccionados
        $this->selectedProducts = $promotion->productos->pluck('id')->toArray();
        $this->selectedCategories = $promotion->categorias->pluck('id')->toArray();

        // Cargar monto mínimo si existe
        $minAmountCondition = $promotion->conditions()->where('type', 'min_amount')->first();
        $this->min_amount = $minAmountCondition?->value;
    }

    public function save()
    {
        $this->validate();

        $this->promotion->update([
            'name' => $this->name,
            'description' => $this->description,
            'discount_type' => $this->discount_type,
            'discount_value' => $this->discount_type !== 'bxgy' ? $this->discount_value : null,
            'buy_qty' => $this->discount_type === 'bxgy' ? $this->buy_qty : null,
            'get_qty' => $this->discount_type === 'bxgy' ? $this->get_qty : null,
            'starts_at' => $this->starts_at ?: null,
            'ends_at' => $this->ends_at ?: null,
            'max_uses' => $this->max_uses ?: null,
            'max_uses_per_user' => $this->max_uses_per_user ?: null,
            'is_active' => $this->is_active,
            'is_automatic' => $this->is_automatic,
            'stackable' => $this->stackable,
            'priority' => $this->priority,
        ]);

        // Sincronizar productos
        $this->promotion->productos()->sync($this->selectedProducts);

        // Sincronizar categorías
        $this->promotion->categorias()->sync($this->selectedCategories);

        // Actualizar condición de monto mínimo
        $this->promotion->conditions()->where('type', 'min_amount')->delete();
        if ($this->min_amount) {
            $this->promotion->conditions()->create([
                'type' => 'min_amount',
                'operator' => '>=',
                'value' => $this->min_amount,
            ]);
        }

        $this->dispatch('notify', ['message' => 'Promoción actualizada correctamente.', 'type' => 'success']);

        return redirect()->route('admin.promociones.index');
    }

    public function render()
    {
        $empresaId = auth()->user()->empresa_id;

        $productos = Producto::where('empresa_id', $empresaId)
            ->where('activo', true)
            ->orderBy('nombre')
            ->get(['id', 'nombre', 'sku']);

        $categorias = Categoria::where('empresa_id', $empresaId)
            ->where('activo', true)
            ->orderBy('nombre')
            ->get(['id', 'nombre']);

        return view('livewire.admin.promociones.edit', [
            'productos' => $productos,
            'categorias' => $categorias,
        ])->layout($this->getLayout(), [
            'title' => 'Editar Promoción',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Promociones', 'route' => 'admin.promociones.index'],
                ['name' => 'Editar', 'active' => true]
            ]
        ]);
    }
}
