<?php

namespace App\Livewire\Admin\Promociones;

use App\Models\Promotion;
use App\Models\Producto;
use App\Models\Categoria;
use Livewire\Component;
use App\Traits\HasDynamicLayout;

class Create extends Component
{
    use HasDynamicLayout;

    public $name = '';
    public $description = '';
    public $discount_type = 'percent';
    public $discount_value = '';
    public $buy_qty = 2;
    public $get_qty = 1;
    public $starts_at = '';
    public $ends_at = '';
    public $max_uses = '';
    public $max_uses_per_user = '';
    public $is_active = true;
    public $is_automatic = false;
    public $stackable = false;
    public $priority = 0;
    public $min_amount = '';
    public $selectedProducts = [];
    public $selectedCategories = [];

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'discount_type' => 'required|in:percent,fixed,bxgy',
            'discount_value' => 'required_unless:discount_type,bxgy|nullable|numeric|min:0',
            'buy_qty' => 'required_if:discount_type,bxgy|nullable|integer|min:1',
            'get_qty' => 'required_if:discount_type,bxgy|nullable|integer|min:1',
            'starts_at' => 'nullable|date',
            'ends_at' => 'nullable|date|after_or_equal:starts_at',
            'max_uses' => 'nullable|integer|min:1',
            'max_uses_per_user' => 'nullable|integer|min:1',
            'is_active' => 'boolean',
            'is_automatic' => 'boolean',
            'stackable' => 'boolean',
            'priority' => 'integer|min:0',
            'min_amount' => 'nullable|numeric|min:0',
        ];
    }

    protected $messages = [
        'name.required' => 'El nombre es obligatorio.',
        'discount_value.required_unless' => 'El valor del descuento es obligatorio.',
        'ends_at.after_or_equal' => 'La fecha de fin debe ser posterior a la fecha de inicio.',
    ];

    public function save()
    {
        $this->validate();

        $empresaId = auth()->user()->empresa_id;

        $promotion = Promotion::create([
            'empresa_id' => $empresaId,
            'name' => $this->name,
            'description' => $this->description,
            'discount_type' => $this->discount_type,
            'discount_value' => $this->discount_type !== 'bxgy' ? $this->discount_value : null,
            'buy_qty' => $this->discount_type === 'bxgy' ? $this->buy_qty : null,
            'get_qty' => $this->discount_type === 'bxgy' ? $this->get_qty : null,
            'starts_at' => $this->starts_at ?: null,
            'ends_at' => $this->ends_at ?: null,
            'max_uses' => $this->max_uses ?: null,
            'max_uses_per_user' => $this->max_uses_per_user ?: null,
            'is_active' => $this->is_active,
            'is_automatic' => $this->is_automatic,
            'stackable' => $this->stackable,
            'priority' => $this->priority,
        ]);

        // Sincronizar productos seleccionados
        if (!empty($this->selectedProducts)) {
            $promotion->productos()->sync($this->selectedProducts);
        }

        // Sincronizar categorías seleccionadas
        if (!empty($this->selectedCategories)) {
            $promotion->categorias()->sync($this->selectedCategories);
        }

        // Crear condición de monto mínimo si existe
        if ($this->min_amount) {
            $promotion->conditions()->create([
                'type' => 'min_amount',
                'operator' => '>=',
                'value' => $this->min_amount,
            ]);
        }

        $this->dispatch('notify', ['message' => 'Promoción creada correctamente.', 'type' => 'success']);

        return redirect()->route('admin.promociones.index');
    }

    public function render()
    {
        $empresaId = auth()->user()->empresa_id;

        $productos = Producto::where('empresa_id', $empresaId)
            ->where('activo', true)
            ->orderBy('nombre')
            ->get(['id', 'nombre', 'sku']);

        $categorias = Categoria::where('empresa_id', $empresaId)
            ->where('activo', true)
            ->orderBy('nombre')
            ->get(['id', 'nombre']);

        return view('livewire.admin.promociones.create', [
            'productos' => $productos,
            'categorias' => $categorias,
        ])->layout($this->getLayout(), [
            'title' => 'Crear Promoción',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Promociones', 'route' => 'admin.promociones.index'],
                ['name' => 'Crear', 'active' => true]
            ]
        ]);
    }
}
