<?php

namespace App\Livewire\Admin\Productos;

use Livewire\Component;
use App\Models\Producto;
use Livewire\WithFileUploads;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\Storage;

class ProductoDetail extends Component
{
    use WithFileUploads, HasDynamicLayout;

    public Producto $producto;
    public $activeTab = 'basic';
    public $variantImages = [];

    protected $listeners = ['refreshProduct' => '$refresh'];

    public function mount(Producto $producto)
    {
        try {
            \Log::info('ProductoDetail: Montando componente', [
                'producto_param' => $producto,
                'producto_tipo' => gettype($producto)
            ]);

            $this->producto = $producto->load([
                'categoria',
                'marca',
                'empresa',
                'images',
                'featuredImage',
                'variants.empresa',
                'inventorySetting',
                'seo'
            ]);

            // Verificar que el producto existe y tiene empresa_id
            if (!$this->producto->empresa_id) {
                abort(404, 'Producto no válido');
            }

            // Verificar que el producto pertenece a la empresa del usuario actual
            if ($this->producto->empresa_id !== auth()->user()->empresa_id) {
                abort(403, 'No tienes permiso para ver este producto');
            }

            \Log::info('ProductoDetail: Producto cargado exitosamente', [
                'producto_id' => $this->producto->id,
                'producto_nombre' => $this->producto->name
            ]);

        } catch (\Exception $e) {
            \Log::error('Error al cargar producto: ' . $e->getMessage());
            abort(404, 'Producto no encontrado');
        }
    }

    public function changeTab($tab)
    {
        $this->activeTab = $tab;
    }

    public function getPriceWithCurrencyProperty()
    {
        return $this->producto->price . ' USD';
    }

    public function getPriceBsWithCurrencyProperty()
    {
        return $this->producto->precio_bs . ' Bs';
    }

    public function getStatusBadgeProperty()
    {
        return $this->producto->status
            ? '<span class="badge bg-success">Activo</span>'
            : '<span class="badge bg-danger">Inactivo</span>';
    }

    public function getStockStatusProperty()
    {
        if ($this->producto->quantity <= 0) {
            return '<span class="badge bg-danger">Sin Stock</span>';
        } elseif ($this->producto->quantity <= $this->producto->quantity_alert) {
            return '<span class="badge bg-warning">Stock Bajo</span>';
        } else {
            return '<span class="badge bg-success">En Stock</span>';
        }
    }

    public function getVariantStockStatus($quantity)
    {
        if ($quantity <= 0) {
            return '<span class="badge bg-danger">Sin Stock</span>';
        } else {
            return '<span class="badge bg-success">En Stock</span>';
        }
    }

    public function getTotalVariantStockProperty()
    {
        return $this->producto->variants->sum('quantity');
    }

    public function getHasVariantsProperty()
    {
        return $this->producto->variants->isNotEmpty();
    }

    public function getInventoryPolicyTextProperty()
    {
        if (!$this->producto->inventorySetting) {
            return 'No configurado';
        }

        $policy = $this->producto->inventorySetting->inventory_policy;
        $track = $this->producto->inventorySetting->track_inventory;
        $backorder = $this->producto->inventorySetting->allow_backorder;

        $text = [];
        $text[] = $track ? 'Rastreo activado' : 'Rastreo desactivado';
        $text[] = $backorder ? 'Backorder permitido' : 'Backorder no permitido';
        $text[] = $policy === 'allow' ? 'Permitir sin stock' : 'Denegar sin stock';

        return implode(' | ', $text);
    }

    public function editVariant($variantId)
    {
        // Redirigir a la edición de la variante
        return redirect()->route('admin.productos.variants.edit', [$this->producto->id, $variantId]);
    }

    public function deleteProduct($productId)
    {
        // Método para eliminar el producto (se puede implementar según necesidad)
        $this->dispatch('show-toast', [
            'type' => 'info',
            'message' => 'Función de eliminación no implementada'
        ]);
    }

    public function render()
    {
        // Verificar si el producto existe
        if (!isset($this->producto) || !$this->producto->id) {
            \Log::error('ProductoDetail: No se encontró el producto');
            return view('livewire.admin.productos.producto-detail-mejorado', [
                'producto' => null
            ])->layout($this->getLayout(), [
                'title' => 'Producto no encontrado',
                'breadcrumb' => [
                    'admin.productos.index' => 'Productos',
                    '#' => 'Producto no encontrado'
                ]
            ]);
        }

        \Log::info('ProductoDetail: Renderizando producto', [
            'producto_id' => $this->producto->id,
            'producto_nombre' => $this->producto->name,
            'empresa_id' => $this->producto->empresa_id,
            'user_empresa_id' => auth()->user()->empresa_id
        ]);

        return view('livewire.admin.productos.producto-detail-mejorado', [
            'producto' => $this->producto
        ])->layout($this->getLayout(), [
            'title' => 'Detalle del Producto: ' . $this->producto->name,
            'breadcrumb' => [
                'admin.productos.index' => 'Productos',
                '#' => 'Detalle del Producto'
            ]
        ]);
    }
}