<?php

namespace App\Livewire\Admin\Productos;

use App\Models\Producto;
use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasDynamicLayout;

class IndexMejorado extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $status = '';
    public $categoria_id = '';
    public $marca_id = '';
    public $stock_filter = '';
    public $perPage = 10;
    public $sortField = 'created_at';
    public $sortDirection = 'desc';

    // Filtros avanzados
    public $showAdvancedFilters = false;
    public $precio_min = '';
    public $precio_max = '';
    public $fecha_desde = '';
    public $fecha_hasta = '';

    // Controles de UI
    public $selectedProductos = [];
    public $selectAll = false;

    // Propiedades para las estadísticas
    public $productoStatsProperty;
    public $salesStatsProperty;

    /**
     * Construye la consulta base de productos con los filtros aplicados.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    private function applyProductFilters($query)
    {
        return $query
            ->when($this->search, function ($q) {
                $q->where(function ($sub) {
                    $sub->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('code', 'like', '%' . $this->search . '%')
                        ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->status !== '', function ($q) {
                $q->where('status', $this->status == '1');
            })
            ->when($this->categoria_id !== '', function ($q) {
                $q->where('categoria_id', $this->categoria_id);
            })
            ->when($this->marca_id !== '', function ($q) {
                $q->where('marca_id', $this->marca_id);
            })
            ->when($this->stock_filter !== '', function ($q) {
                if ($this->stock_filter == 'low') {
                    $q->whereColumn('quantity', '<=', 'quantity_alert');
                } elseif ($this->stock_filter == 'out') {
                    $q->where('quantity', 0);
                } elseif ($this->stock_filter == 'in') {
                    $q->where('quantity', '>', 0);
                }
            })
            ->when($this->precio_min !== '', function ($q) {
                $q->where('price', '>=', $this->precio_min);
            })
            ->when($this->precio_max !== '', function ($q) {
                $q->where('price', '<=', $this->precio_max);
            })
            ->when($this->fecha_desde !== '', function ($q) {
                $q->whereDate('created_at', '>=', $this->fecha_desde);
            })
            ->when($this->fecha_hasta !== '', function ($q) {
                $q->whereDate('created_at', '<=', $this->fecha_hasta);
            });
    }

    protected $queryString = [
        'search' => ['except' => ''],
        'status' => ['except' => ''],
        'categoria_id' => ['except' => ''],
        'marca_id' => ['except' => ''],
        'stock_filter' => ['except' => ''],
        'perPage',
        'sortField',
        'sortDirection'
    ];

    protected $listeners = [
        'refreshProductos' => '$refresh',
        'deleteSelected' => 'deleteSelected',
        'exportSelected' => 'exportSelected'
    ];

    public function mount()
    {
        $this->loadStats();
        $this->loadSalesStats();
    }

    public function updatedSearch()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedStatus()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedSelectedProductos()
    {
        $query = Producto::query();
        $query = $this->applyProductFilters($query);
        $productos = $query->orderBy($this->sortField, $this->sortDirection)->get();

        $this->selectAll = count($this->selectedProductos) === $productos->count();
    }

    public function updatedSelectAll()
    {
        if ($this->selectAll) {
            $query = Producto::query();
            $query = $this->applyProductFilters($query);
            $productos = $query->orderBy($this->sortField, $this->sortDirection)->get();

            $this->selectedProductos = $productos->pluck('id')->toArray();
        } else {
            $this->selectedProductos = [];
        }
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function loadStats()
    {
        $query = Producto::query();
        $query = $this->applyProductFilters($query);

        $total = $query->count();
        $activos = $query->where('status', true)->count();
        $inactivos = $query->where('status', false)->count();
        $bajoStock = $query->whereColumn('quantity', '<=', 'quantity_alert')->count();
        $sinStock = $query->where('quantity', 0)->count();

        // Calcular porcentajes de cambio (simulados)
        $cambioTotal = '+12.5%';
        $cambioActivos = '+8.3%';
        $cambioBajoStock = '-15.2%';

        $this->productoStatsProperty = [
            'total' => $total,
            'activos' => $activos,
            'inactivos' => $inactivos,
            'bajo_stock' => $bajoStock,
            'sin_stock' => $sinStock,
            'cambio_total' => $cambioTotal,
            'cambio_activos' => $cambioActivos,
            'cambio_bajo_stock' => $cambioBajoStock
        ];
    }

    public function loadSalesStats()
    {
        // Estadísticas de ventas (simuladas por ahora)
        $this->salesStatsProperty = [
            'ventas_totales' => '$45,678.90',
            'ordenes_totales' => 1234,
            'cambio_ventas' => '+5.7%',
            'cambio_ordenes' => '+12.4%'
        ];
    }

    public function delete($id)
    {
        $producto = Producto::findOrFail($id);

        if ($producto->detallesPedido()->count() > 0) {
            $this->dispatch('notify', [
                'type' => 'error',
                'message' => 'No se puede eliminar el producto porque tiene pedidos asociados.'
            ]);
            return;
        }

        $producto->delete();

        $this->dispatch('notify', [
            'type' => 'success',
            'message' => 'Producto eliminado correctamente.'
        ]);

        $this->loadStats();
        $this->reset(['selectedProductos', 'selectAll']);
    }

    public function deleteSelected()
    {
        if (empty($this->selectedProductos)) {
            $this->dispatch('notify', [
                'type' => 'warning',
                'message' => 'Por favor selecciona al menos un producto.'
            ]);
            return;
        }

        try {
            \DB::transaction(function () {
                $productos = Producto::whereIn('id', $this->selectedProductos)
                    ->withCount('detallesPedido')
                    ->get();

                $eliminados = 0;
                foreach ($productos as $producto) {
                    if ($producto->detalles_pedido_count === 0) {
                        $producto->delete();
                        $eliminados++;
                    }
                }

                if ($eliminados > 0) {
                    $this->dispatch('notify', [
                        'type' => 'success',
                        'message' => "$eliminados productos eliminados correctamente."
                    ]);
                } else {
                    $this->dispatch('notify', [
                        'type' => 'error',
                        'message' => 'No se pudieron eliminar los productos seleccionados porque tienen pedidos asociados.'
                    ]);
                }
            });
        } catch (\Exception $e) {
            $this->dispatch('notify', [
                'type' => 'error',
                'message' => 'Error al eliminar productos: ' . $e->getMessage()
            ]);
        }

        $this->loadStats();
        $this->reset(['selectedProductos', 'selectAll']);
    }

    public function exportSelected()
    {
        if (empty($this->selectedProductos)) {
            $this->dispatch('notify', [
                'type' => 'warning',
                'message' => 'Por favor selecciona al menos un producto.'
            ]);
            return;
        }

        // Aquí iría la lógica de exportación
        $this->dispatch('notify', [
            'type' => 'info',
            'message' => 'Exportación en proceso...'
        ]);
    }

    public function duplicate($id)
    {
        $producto = Producto::findOrFail($id);
        $nuevoProducto = $producto->replicate();
        $nuevoProducto->code = $producto->code . '-COPY-' . time();
        $nuevoProducto->name = $producto->name . ' (Copia)';
        $nuevoProducto->save();

        $this->dispatch('notify', [
            'type' => 'success',
            'message' => 'Producto duplicado correctamente.'
        ]);

        $this->loadStats();
    }

    public function toggleStatus($id)
    {
        $producto = Producto::findOrFail($id);
        $producto->status = !$producto->status;
        $producto->save();

        $this->dispatch('notify', [
            'type' => 'success',
            'message' => 'Estado del producto actualizado correctamente.'
        ]);

        $this->loadStats();
    }

    public function resetFilters()
    {
        $this->reset([
            'search', 'status', 'categoria_id', 'marca_id', 'stock_filter',
            'precio_min', 'precio_max', 'fecha_desde', 'fecha_hasta'
        ]);
        $this->resetPage();
        $this->loadStats();
    }

    public function render()
    {
        $query = Producto::with(['categoria', 'marca', 'images']);

        $query = $this->applyProductFilters($query);

        $productos = $query->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        // Cargar categorías y marcas para los filtros
        $categorias = \App\Models\Categoria::where('activo', true)
            ->orderBy('nombre')
            ->get();

        $marcas = \App\Models\Marca::where('activo', true)
            ->orderBy('nombre')
            ->get();

        return $this->renderWithLayout('livewire.admin.productos.index-mejorado', [
            'productos' => $productos,
            'stats' => $this->productoStatsProperty,
            'salesStats' => $this->salesStatsProperty,
            'categorias' => $categorias,
            'marcas' => $marcas
        ], [
            'title' => 'Gestión de Productos',
            'description' => 'Listado mejorado de productos con estadísticas avanzadas y filtros',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Productos', 'active' => true]
            ]
        ]);
    }
}
