<?php

namespace App\Livewire\Admin\Pos;

use App\Models\Pedido;
use App\Models\Caja;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;

class Reportes extends Component
{
    use WithPagination, HasDynamicLayout, HasRegionalFormatting;

    public $fechaInicio;
    public $fechaFin;
    public $cajaId = '';
    public $metodoPago = '';
    public $tipoDocumento = '';

    public function mount()
    {
        $this->fechaInicio = Carbon::today()->format('Y-m-d');
        $this->fechaFin = Carbon::today()->format('Y-m-d');
    }

    public function getVentasProperty()
    {
        $query = Pedido::with(['detalles.producto', 'pagos', 'serie', 'caja'])
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('tipo_venta', 'pos')
            ->whereBetween('created_at', [
                $this->fechaInicio . ' 00:00:00',
                $this->fechaFin . ' 23:59:59'
            ]);

        if ($this->cajaId) {
            $query->where('caja_id', $this->cajaId);
        }

        if ($this->metodoPago) {
            $query->whereHas('pagos', function($q) {
                $q->where('metodo_pago', $this->metodoPago);
            });
        }

        if ($this->tipoDocumento) {
            $query->where('tipo_documento', $this->tipoDocumento);
        }

        return $query->orderBy('created_at', 'desc')->paginate(20);
    }

    public function getCajasProperty()
    {
        return Caja::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function getResumenProperty()
    {
        $ventas = $this->ventas;
        
        return [
            'total_ventas' => $ventas->count(),
            'monto_total' => $ventas->sum('total'),
            'promedio_venta' => $ventas->count() > 0 ? $ventas->sum('total') / $ventas->count() : 0,
            'por_metodo_pago' => $this->getVentasPorMetodoPago(),
            'por_tipo_documento' => $this->getVentasPorTipoDocumento(),
        ];
    }

    private function getVentasPorMetodoPago()
    {
        return Pedido::join('pedido_pagos', 'pedidos.id', '=', 'pedido_pagos.pedido_id')
            ->where('pedidos.empresa_id', auth()->user()->empresa_id)
            ->where('pedidos.tipo_venta', 'pos')
            ->whereBetween('pedidos.created_at', [
                $this->fechaInicio . ' 00:00:00',
                $this->fechaFin . ' 23:59:59'
            ])
            ->selectRaw('pedido_pagos.metodo_pago, COUNT(*) as cantidad, SUM(pedido_pagos.monto) as total')
            ->groupBy('pedido_pagos.metodo_pago')
            ->get();
    }

    private function getVentasPorTipoDocumento()
    {
        return Pedido::where('empresa_id', auth()->user()->empresa_id)
            ->where('tipo_venta', 'pos')
            ->whereBetween('created_at', [
                $this->fechaInicio . ' 00:00:00',
                $this->fechaFin . ' 23:59:59'
            ])
            ->selectRaw('tipo_documento, COUNT(*) as cantidad, SUM(total) as total')
            ->groupBy('tipo_documento')
            ->get();
    }

    public function exportar()
    {
        $ventas = $this->ventas;
        
        $filename = 'ventas_pos_' . $this->fechaInicio . '_' . $this->fechaFin . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($ventas) {
            $file = fopen('php://output', 'w');
            
            // Headers
            fputcsv($file, [
                'Fecha',
                'Documento',
                'Cliente',
                'Total',
                'Método Pago',
                'Estado',
                'Caja'
            ]);

            foreach ($ventas as $venta) {
                fputcsv($file, [
                    $venta->created_at->format('d/m/Y H:i'),
                    $venta->serie->serie . '-' . str_pad($venta->numero_documento, 8, '0', STR_PAD_LEFT),
                    $venta->user->name ?? 'Cliente General',
                    $venta->total,
                    $venta->pagos->first()->metodo_pago ?? '',
                    $venta->estado,
                    $venta->caja->nombre ?? ''
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function render()
    {
        return view('livewire.admin.pos.reportes', [
            'ventas' => $this->ventas,
            'cajas' => $this->cajas,
            'resumen' => $this->resumen
        ])->layout($this->getLayout());
    }
}