<?php

namespace App\Livewire\Admin\Pos;

use App\Models\CajaMovimiento;
use App\Models\User;
use App\Services\CajaMovimientoService;
use Livewire\Component;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;

class Aprobaciones extends Component
{
    use HasDynamicLayout, HasRegionalFormatting;

    // Propiedades de la vista

    public $movimientosPendientes = [];
    public $motivoRechazo = '';
    public $movimientoSeleccionado = null;
    public $showModalRechazo = false;

    protected $rules = [
        'motivoRechazo' => 'required|string|max:500',
    ];

    public function mount()
    {
        $this->cargarMovimientosPendientes();
    }

    public function cargarMovimientosPendientes()
    {
        $empresaId = auth()->user()->empresa_id;
        $sucursalId = auth()->user()->sucursal_id;

        $this->movimientosPendientes = CajaMovimiento::whereHas('caja', function ($query) use ($empresaId, $sucursalId) {
            $query->where('empresa_id', $empresaId)
                  ->where('sucursal_id', $sucursalId);
        })
        ->where('estado_aprobacion', 'pendiente')
        ->with(['usuario', 'caja'])
        ->orderBy('created_at', 'desc')
        ->get()
        ->map(function ($movimiento) {
            return [
                'id' => $movimiento->id,
                'tipo' => $movimiento->tipo,
                'tipo_salida' => $movimiento->tipo_salida,
                'monto' => $movimiento->monto,
                'motivo' => $movimiento->motivo,
                'fecha' => $movimiento->created_at->format('d/m/Y H:i'),
                'usuario' => $movimiento->usuario->name,
                'ip_address' => $movimiento->ip_address,
                'caja_id' => $movimiento->caja_id,
            ];
        })
        ->toArray();
    }

    public function aprobarMovimiento($movimientoId)
    {
        $movimiento = CajaMovimiento::find($movimientoId);

        if (!$movimiento) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Movimiento no encontrado.'
            ]);
            return;
        }

        $service = new CajaMovimientoService();
        $resultado = $service->aprobarMovimiento($movimiento, auth()->user());

        if (!$resultado['success']) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => $resultado['message']
            ]);
            return;
        }

        $this->dispatch('show-toast', [
            'type' => 'success',
            'message' => 'Movimiento aprobado exitosamente.'
        ]);

        $this->cargarMovimientosPendientes();
    }

    public function abrirModalRechazo($movimientoId)
    {
        $this->movimientoSeleccionado = $movimientoId;
        $this->motivoRechazo = '';
        $this->showModalRechazo = true;
    }

    public function cerrarModalRechazo()
    {
        $this->showModalRechazo = false;
        $this->movimientoSeleccionado = null;
        $this->motivoRechazo = '';
    }

    public function rechazarMovimiento()
    {
        $this->validate();

        $movimiento = CajaMovimiento::find($this->movimientoSeleccionado);

        if (!$movimiento) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Movimiento no encontrado.'
            ]);
            return;
        }

        $service = new CajaMovimientoService();
        $resultado = $service->rechazarMovimiento($movimiento, auth()->user(), $this->motivoRechazo);

        if (!$resultado['success']) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => $resultado['message']
            ]);
            return;
        }

        $this->dispatch('show-toast', [
            'type' => 'success',
            'message' => 'Movimiento rechazado exitosamente.'
        ]);

        $this->cerrarModalRechazo();
        $this->cargarMovimientosPendientes();
    }

    public function render()
    {
        return view('livewire.admin.pos.aprobaciones')
            ->layout($this->getLayout());
    }
}
