<?php

namespace App\Livewire\Admin\Pedidos;

use App\Models\Pedido;
use App\Models\Empleado;
use App\Services\PedidoService;
use App\Traits\HasDynamicLayout;
use Livewire\Component;

class Show extends Component
{
    use HasDynamicLayout;

    public Pedido $pedido;
    public $showAsignarModal = false;
    public $showCancelarModal = false;
    public $empleadoSeleccionado = null;
    public $motivoCancelacion = '';

    /**
     * Get icon class for payment method
     */
    public function getPaymentMethodIcon($method): string
    {
        return match(strtolower($method)) {
            'transferencia' => 'ri-bank-line',
            'pago_movil' => 'ri-smartphone-line',
            'efectivo' => 'ri-money-dollar-circle-line',
            'tarjeta' => 'ri-bank-card-line',
            'zelle' => 'ri-exchange-funds-line',
            default => 'ri-secure-payment-line'
        };
    }

    /**
     * Get icon class for order status
     */
    public function getStatusIcon($status): string
    {
        return match(strtolower($status)) {
            'pendiente' => 'ri-time-line',
            'pagado' => 'ri-check-line',
            'asignado' => 'ri-user-line',
            'entregado' => 'ri-check-double-line',
            'cancelado' => 'ri-close-line',
            default => 'ri-information-line'
        };
    }

    /**
     * Get badge class for order status
     */
    public function getStatusBadgeClass($status): string
    {
        return match(strtolower($status)) {
            'pendiente' => 'bg-warning',
            'pagado' => 'bg-success',
            'asignado' => 'bg-info',
            'entregado' => 'bg-primary',
            'cancelado' => 'bg-danger',
            default => 'bg-secondary'
        };
    }

    /**
     * Get timeline steps with icons for order tracking
     */
    public function getTimelineSteps(): array
    {
        return [
            'pendiente' => [
                'icon' => 'ri-time-line',
                'label' => 'Pendiente',
                'description' => 'Pedido recibido, esperando confirmación de pago'
            ],
            'pagado' => [
                'icon' => 'ri-check-line',
                'label' => 'Pagado',
                'description' => 'Pago confirmado, preparando pedido'
            ],
            'asignado' => [
                'icon' => 'ri-user-line',
                'label' => 'Asignado',
                'description' => 'Empleado asignado para entrega'
            ],
            'entregado' => [
                'icon' => 'ri-check-double-line',
                'label' => 'Entregado',
                'description' => 'Pedido entregado exitosamente'
            ],
            'cancelado' => [
                'icon' => 'ri-close-line',
                'label' => 'Cancelado',
                'description' => 'Pedido cancelado'
            ]
        ];
    }

    public function mount(Pedido $pedido)
    {
        $this->pedido = $pedido->load([
            'user',
            'detalles' => function($query) {
                $query->with('producto');
            },
            'empleados',
            'referenciaBancaria',
            'pagos',
            'historial' => function($query) {
                $query->orderBy('created_at', 'desc');
            }
        ]);

        // Debug: verificar si los productos existen
        foreach($this->pedido->detalles as $detalle) {
            if (!$detalle->producto) {
                \Log::info('Producto no encontrado para detalle', [
                    'detalle_id' => $detalle->id,
                    'producto_id' => $detalle->productoId
                ]);
            }
        }
    }

    public function confirmarPago()
    {
        $pedidoService = app(PedidoService::class);
        
        // Cargar relaciones necesarias para la notificación
        $this->pedido->load(['user', 'detalles.producto', 'empleados']);
        
        $pedidoService->confirmarPago($this->pedido);

        $this->pedido->refresh();
        session()->flash('message', 'Pago confirmado y notificación enviada al cliente.');
    }

    public function abrirModalAsignar()
    {
        $this->empleadoSeleccionado = null;
        $this->showAsignarModal = true;
    }

    public function asignarEmpleado()
    {
        $this->validate([
            'empleadoSeleccionado' => 'required|exists:empleados,id'
        ]);

        $empleado = Empleado::findOrFail($this->empleadoSeleccionado);
        $pedidoService = app(PedidoService::class);

        // Cargar relaciones necesarias para la notificación
        $this->pedido->load(['user', 'detalles.producto', 'empleados']);

        $pedidoService->asignarEmpleado($this->pedido, $empleado);

        $this->showAsignarModal = false;
        $this->pedido->refresh();
        session()->flash('message', 'Pedido asignado y notificación enviada al empleado.');
    }

    public function abrirModalCancelar()
    {
        $this->motivoCancelacion = '';
        $this->showCancelarModal = true;
    }

    public function cancelarPedido()
    {
        $this->validate([
            'motivoCancelacion' => 'required|string|min:10'
        ]);

        $pedidoService = app(PedidoService::class);
        $pedidoService->cancelarPedido($this->pedido, $this->motivoCancelacion);

        $this->showCancelarModal = false;
        $this->pedido->refresh();
        session()->flash('message', 'Pedido cancelado y notificación enviada al cliente.');
    }

    public function finalizarEntrega()
    {
        $pedidoService = app(PedidoService::class);
        
        // Cargar relaciones necesarias para la notificación
        $this->pedido->load(['user', 'detalles.producto', 'empleados']);
        
        $pedidoService->finalizarEntrega($this->pedido);

        $this->pedido->refresh();
        session()->flash('message', 'Entrega finalizada exitosamente.');
    }

    public function render()
    {
        $empleados = Empleado::where('status', true)->get();

        return $this->renderWithLayout('livewire.admin.pedidos.show', compact('empleados'));
    }
}