<?php

namespace App\Livewire\Admin\Pedidos;

use Livewire\Component;
use App\Models\Pedido;
use App\Models\Producto;
use App\Models\User;
use App\Traits\HasDynamicLayout;

class Edit extends Component
{
    use HasDynamicLayout;

    public $pedido;
    public $fecha_entrega = '';
    public $notas = '';
    
    // Productos en el carrito
    public $cart = [];
    public $selectedProduct = '';
    public $quantity = 1;

    // Búsqueda de productos
    public $searchTerm = '';
    public $searchResults = [];
    public $showSearchResults = false;

    protected $rules = [
        'fecha_entrega' => 'nullable|date|after_or_equal:today',
        'notas' => 'nullable|string|max:1000',
        'quantity' => 'integer|min:1|max:1000',
    ];

    protected $messages = [
        'fecha_entrega.after_or_equal' => 'La fecha de entrega debe ser hoy o posterior.',
        'quantity.min' => 'La cantidad mínima es 1.',
        'quantity.max' => 'La cantidad máxima es 1000.',
    ];

    public function mount($pedidoId)
    {
        $this->pedido = Pedido::with(['detalles.producto', 'user'])->findOrFail($pedidoId);
        
        // Cargar datos del pedido
        $this->fecha_entrega = $this->pedido->fecha_entrega ? $this->pedido->fecha_entrega->format('Y-m-d') : '';
        $this->notas = $this->pedido->notas;
        
        // Cargar productos existentes en el carrito
        $this->loadExistingProducts();
    }

    public function render()
    {
        return view('livewire.admin.pedidos.edit')
            ->layout($this->getLayout(), [
                'title' => 'Editar Pedido #' . $this->pedido->codigo,
                'breadcrumb' => [
                    ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                    ['name' => 'Pedidos', 'route' => 'admin.pedidos.index'],
                    ['name' => 'Editar Pedido', 'active' => true]
                ]
            ]);
    }

    public function loadExistingProducts()
    {
        $this->cart = [];
        foreach ($this->pedido->detalles as $detalle) {
            $this->cart[] = [
                'detalle_id' => $detalle->id,
                'producto_id' => $detalle->producto->id,
                'nombre' => $detalle->producto->name,
                'codigo' => $detalle->producto->code,
                'cantidad' => $detalle->quantity,
                'precio_unitario' => (float) $detalle->price,
                'subtotal' => (float) $detalle->total_usd,
                'is_existing' => true
            ];
        }
    }

    public function searchProducts()
    {
        if (strlen($this->searchTerm) >= 2) {
            $query = Producto::where('status', true);
            
            // Buscar por código exacto primero
            $exactMatch = clone $query;
            $productos = $exactMatch->where('code', $this->searchTerm)->limit(5)->get();
            
            // Si no hay coincidencia exacta, buscar por coincidencias parciales
            if ($productos->isEmpty()) {
                $productos = $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('description', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('code', 'like', '%' . $this->searchTerm . '%');
                })
                ->limit(8)
                ->get();
            }
            
            $this->searchResults = [];
            foreach ($productos as $producto) {
                $this->searchResults[] = [
                    'id' => $producto->id,
                    'name' => $producto->name,
                    'code' => $producto->code,
                    'description' => $producto->description,
                    'price' => (float) $producto->price,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'categoria_name' => $producto->categoria ? $producto->categoria->name : null,
                ];
            }
            
            $this->showSearchResults = true;
        } else {
            $this->searchResults = [];
            $this->showSearchResults = false;
        }
    }

    public function updatedSearchTerm()
    {
        $this->searchProducts();
    }

    public function selectProduct($productId)
    {
        $this->selectedProduct = $productId;
        $this->searchTerm = '';
        $this->showSearchResults = false;

        $producto = Producto::find($productId);
        if ($producto) {
            $this->searchTerm = $producto->name;
        }
    }

    public function hideSearchResults()
    {
        $this->showSearchResults = false;
    }

    public function addProductToCart()
    {
        $this->validate([
            'quantity' => 'required|integer|min:1|max:1000'
        ]);

        if (!$this->selectedProduct || $this->quantity < 1) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Seleccione un producto y una cantidad válida.'
            ]);
            return;
        }

        $producto = Producto::find($this->selectedProduct);

        if (!$producto) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Producto no encontrado.'
            ]);
            return;
        }

        // Verificar stock disponible (considerando productos ya en el carrito)
        $cantidadEnCarrito = collect($this->cart)
            ->where('producto_id', $producto->id)
            ->sum('cantidad');
        
        $stockDisponible = $producto->quantity + $cantidadEnCarrito;

        if ($stockDisponible < $this->quantity) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'No hay suficiente stock disponible. Stock actual: ' . $stockDisponible
            ]);
            return;
        }

        // Verificar si el producto ya está en el carrito
        $existingItemIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id;
        });

        if ($existingItemIndex !== false) {
            // Actualizar cantidad si ya existe
            $newQuantity = $this->cart[$existingItemIndex]['cantidad'] + $this->quantity;
            $this->cart[$existingItemIndex]['cantidad'] = $newQuantity;
            $this->cart[$existingItemIndex]['subtotal'] = $newQuantity * $this->cart[$existingItemIndex]['precio_unitario'];
        } else {
            // Agregar nuevo producto al carrito
            $this->cart[] = [
                'detalle_id' => null,
                'producto_id' => $producto->id,
                'nombre' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $this->quantity,
                'precio_unitario' => (float) $producto->price,
                'subtotal' => $this->quantity * (float) $producto->price,
                'is_existing' => false
            ];
        }

        // Resetear campos
        $this->selectedProduct = '';
        $this->quantity = 1;
        $this->searchTerm = '';
        $this->showSearchResults = false;

        $this->dispatch('product-added');
        $this->dispatch('show-toast', [
            'type' => 'success',
            'message' => 'Producto agregado al carrito correctamente.'
        ]);
    }

    public function updateQuantity($index, $newQuantity)
    {
        if ($newQuantity < 1) {
            return;
        }

        $item = $this->cart[$index];
        $producto = Producto::find($item['producto_id']);
        
        if (!$producto) {
            return;
        }

        // Calcular stock disponible
        $cantidadOriginal = $item['is_existing'] ? 
            $this->pedido->detalles->where('productoId', $item['producto_id'])->first()->quantity ?? 0 : 0;
        
        $stockDisponible = $producto->quantity + $cantidadOriginal;

        if ($newQuantity > $stockDisponible) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Stock insuficiente. Disponible: ' . $stockDisponible
            ]);
            return;
        }

        $this->cart[$index]['cantidad'] = $newQuantity;
        $this->cart[$index]['subtotal'] = $newQuantity * $this->cart[$index]['precio_unitario'];

        $this->dispatch('show-toast', [
            'type' => 'info',
            'message' => 'Cantidad actualizada.'
        ]);
    }

    public function removeFromCart($index)
    {
        if (isset($this->cart[$index])) {
            $productName = $this->cart[$index]['nombre'];
            unset($this->cart[$index]);
            $this->cart = array_values($this->cart);
            
            $this->dispatch('product-removed');
            $this->dispatch('show-toast', [
                'type' => 'info',
                'message' => 'Producto "' . $productName . '" eliminado del carrito.'
            ]);
        }
    }

    public function getTotalProperty()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function save()
    {
        if (empty($this->cart)) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Debe tener al menos un producto en el pedido.'
            ]);
            return;
        }

        $this->validate();

        try {
            \DB::beginTransaction();

            // Actualizar datos del pedido
            $this->pedido->update([
                'fecha_entrega' => $this->fecha_entrega ?: null,
                'notas' => $this->notas,
                'total_usd' => $this->total,
            ]);

            // Eliminar detalles existentes que no están en el carrito
            $existingDetalleIds = collect($this->cart)
                ->where('is_existing', true)
                ->pluck('detalle_id')
                ->filter()
                ->toArray();

            $this->pedido->detalles()
                ->whereNotIn('id', $existingDetalleIds)
                ->each(function($detalle) {
                    // Restaurar stock del producto eliminado
                    $producto = Producto::find($detalle->productoId);
                    if ($producto) {
                        $producto->increment('quantity', $detalle->quantity);
                    }
                    $detalle->delete();
                });

            // Procesar productos en el carrito
            foreach ($this->cart as $item) {
                if ($item['is_existing'] && $item['detalle_id']) {
                    // Actualizar detalle existente
                    $detalle = $this->pedido->detalles()->find($item['detalle_id']);
                    if ($detalle) {
                        $cantidadAnterior = $detalle->quantity;
                        $diferencia = $item['cantidad'] - $cantidadAnterior;
                        
                        $detalle->update([
                            'quantity' => $item['cantidad'],
                            'price' => $item['precio_unitario'],
                            'total_usd' => $item['subtotal'],
                        ]);

                        // Ajustar stock
                        if ($diferencia != 0) {
                            $producto = Producto::find($item['producto_id']);
                            if ($producto) {
                                $producto->decrement('quantity', $diferencia);
                            }
                        }
                    }
                } else {
                    // Crear nuevo detalle
                    $this->pedido->detalles()->create([
                        'productoId' => $item['producto_id'],
                        'quantity' => $item['cantidad'],
                        'price' => $item['precio_unitario'],
                        'total_usd' => $item['subtotal'],
                        'empresaId' => auth()->user()->empresa_id ?? null,
                        'userId' => auth()->user()->id,
                    ]);

                    // Reducir stock
                    $producto = Producto::find($item['producto_id']);
                    if ($producto) {
                        $producto->decrement('quantity', $item['cantidad']);
                    }
                }
            }

            // Recalcular total del pedido
            $nuevoTotal = $this->pedido->detalles()->sum('total_usd');
            $tasaCambio = \App\Models\ExchangeRate::whereDate('date', today())
                ->orderBy('fetch_time', 'desc')
                ->first();
            $usdRate = $tasaCambio ? $tasaCambio->usd_rate : 36.5;

            $this->pedido->update([
                'total_usd' => $nuevoTotal,
                'total_bs' => $nuevoTotal * $usdRate,
            ]);

            \DB::commit();

            session()->flash('message', 'Pedido #' . $this->pedido->codigo . ' actualizado correctamente.');
            session()->flash('message_type', 'success');

            return redirect()->route('admin.pedidos.show', $this->pedido);
            
        } catch (\Exception $e) {
            \DB::rollBack();
            
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al actualizar el pedido: ' . $e->getMessage()
            ]);
        }
    }
}