<?php

namespace App\Livewire\Admin\NotasVenta;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\NotaVenta;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDynamicLayout;

class Index extends Component
{
    use WithPagination, HasRegionalFormatting, HasDynamicLayout;

    public $search = '';
    public $perPage = 15;
    public $fechaInicio = '';
    public $fechaFin = '';
    public $estado = '';
    public $orderBy = 'created_at';
    public $orderDirection = 'desc';

    protected $paginationTheme = 'bootstrap';

    protected $queryString = [
        'search' => ['except' => ''],
        'fechaInicio' => ['except' => ''],
        'fechaFin' => ['except' => ''],
        'estado' => ['except' => ''],
        'orderBy' => ['except' => 'created_at'],
        'orderDirection' => ['except' => 'desc'],
    ];

    public function mount()
    {
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function limpiarFiltros()
    {
        $this->search = '';
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');
        $this->estado = '';
        $this->resetPage();
    }

    public function orderBy($field)
    {
        if ($this->orderBy === $field) {
            $this->orderDirection = $this->orderDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->orderBy = $field;
            $this->orderDirection = 'asc';
        }
    }

    public function getTotalNotasProperty()
    {
        return NotaVenta::whereDate('created_at', '>=', $this->fechaInicio)
            ->whereDate('created_at', '<=', $this->fechaFin)
            ->count();
    }

    public function getTotalMontoProperty()
    {
        return NotaVenta::whereDate('created_at', '>=', $this->fechaInicio)
            ->whereDate('created_at', '<=', $this->fechaFin)
            ->sum('total');
    }

    public function render()
    {
        $query = NotaVenta::with(['cliente', 'user', 'cotizaciones', 'venta'])
            ->when($this->search, function ($query) {
                $query->where(function($q) {
                    $q->where('numero', 'like', '%' . $this->search . '%')
                      ->orWhere('serie', 'like', '%' . $this->search . '%')
                      ->orWhereRaw("CONCAT(serie, '-', LPAD(numero, 8, '0')) LIKE ?", ['%' . $this->search . '%'])
                      ->orWhereHas('cliente', function ($q) {
                          $q->where('nombres', 'like', '%' . $this->search . '%')
                            ->orWhere('apellidos', 'like', '%' . $this->search . '%')
                            ->orWhere('documento_identidad', 'like', '%' . $this->search . '%');
                      })
                      ->orWhereHas('user', function ($q) {
                          $q->where('name', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->fechaInicio, function ($query) {
                $query->whereDate('fecha_emision', '>=', $this->fechaInicio);
            })
            ->when($this->fechaFin, function ($query) {
                $query->whereDate('fecha_emision', '<=', $this->fechaFin);
            })
            ->when($this->estado, function ($query) {
                $query->where('estado', $this->estado);
            })
            ->orderBy($this->orderBy, $this->orderDirection);

        $notasVenta = $query->paginate($this->perPage);
        $estados = [
            'pendiente' => 'Pendiente',
            'procesada' => 'Procesada',
            'anulada' => 'Anulada'
        ];

        return view('livewire.admin.notas-venta.index', [
            'notasVenta' => $notasVenta,
            'estados' => $estados
        ])->layout($this->getLayout());
    }
}
