<?php

namespace App\Livewire\Admin\NotasVenta;

use App\Models\NotaVenta;
use App\Models\Producto;
use App\Models\User;
use App\Models\Caja;
use App\Models\Serie;
use App\Models\Cliente;
use Livewire\Component;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;

class Edit extends Component
{
    use HasDynamicLayout, HasRegionalFormatting;

    public $notaVenta;
    public $codigo;
    public $cliente_id = null;
    public $metodo_pago = 'efectivo';
    public $serie_id = null;
    public $observaciones = '';

    public $cart = [];
    public $search = '';
    public $searchResults = [];

    // Modal de variantes
    public $showVariantsModal = false;
    public $selectedProductForVariants = null;
    public $selectedVariant = null;
    public $variantQuantity = 1;

    // Modal de crear cliente
    public $showCreateClientModal = false;
    public $newClient = [
        'nombres' => '',
        'apellidos' => '',
        'email' => '',
        'telefono' => '',
        'direccion' => '',
        'documento_identidad' => '',
    ];

    public $clientes = [];
    public $series = [];
    public $productos = [];

    protected $rules = [
        //'metodo_pago' => 'required|in:efectivo,transferencia,tarjeta,pago_movil,punto_de_venta',
        'cart' => 'required|array|min:1',
        'cart.*.producto_id' => 'required|exists:productos,id',
        'cart.*.cantidad' => 'required|integer|min:1',
        'cart.*.precio_unitario' => 'required|numeric|min:0',
    ];

    protected $clientRules = [
        'newClient.nombres' => 'required|string|max:255',
        'newClient.apellidos' => 'required|string|max:255',
        'newClient.email' => 'nullable|email|max:255',
        'newClient.telefono' => 'required|string|max:20',
        'newClient.direccion' => 'nullable|string|max:500',
        'newClient.documento_identidad' => 'nullable|string|max:50',
    ];

    public function mount(NotaVenta $notaVenta)
    {
        // Inicializar configuración regional
        $this->bootHasRegionalFormatting();

        // Cargar la nota de venta con sus relaciones
        $this->notaVenta = $notaVenta->load(['cotizaciones', 'venta']);
        $this->codigo = $notaVenta->codigo;
        $this->cliente_id = $notaVenta->cliente_id;
        $this->metodo_pago = $notaVenta->metodo_pago ?? 'efectivo';
        $this->observaciones = $notaVenta->observaciones;

        // Cargar detalles existentes
        foreach ($notaVenta->detalles as $detalle) {
            $this->cart[] = [
                'producto_id' => $detalle->producto_id,
                'variante_id' => $detalle->producto_variant_id,
                'nombre' => $detalle->descripcion,
                'codigo' => $detalle->producto ? $detalle->producto->code : '',
                'cantidad' => $detalle->cantidad,
                'precio_unitario' => $detalle->precio_unitario,
                'subtotal' => $detalle->subtotal,
            ];
        }

        $this->loadData();
    }

    public function getCorrelativoActualProperty()
    {
        if (!$this->notaVenta) {
            return null;
        }

        return $this->notaVenta->serie . '-' . str_pad($this->notaVenta->numero, 8, '0', STR_PAD_LEFT);
    }

    public function loadData()
    {
        $this->clientes = Cliente::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->orderBy('nombres')
            ->get();

        $this->series = Serie::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('tipo_documento', 'nota_venta')
            ->where('activo', true)
            ->get();
    }

    public function openCreateClientModal()
    {
        $this->showCreateClientModal = true;
    }

    public function closeCreateClientModal()
    {
        $this->showCreateClientModal = false;
        $this->reset('newClient');
    }

    public function createClient()
    {
        $this->validate($this->clientRules);

        try {
            $cliente = Cliente::create([
                'nombres' => $this->newClient['nombres'],
                'apellidos' => $this->newClient['apellidos'],
                'email' => $this->newClient['email'],
                'telefono' => $this->newClient['telefono'],
                'direccion' => $this->newClient['direccion'],
                'documento_identidad' => $this->newClient['documento_identidad'],
                'empresa_id' => auth()->user()->empresa_id,
                'sucursal_id' => auth()->user()->sucursal_id,
            ]);

            // Recargar la lista de clientes
            $this->loadData();

            // Seleccionar el nuevo cliente
            $this->cliente_id = $cliente->id;

            // Forzar actualización del frontend
            $this->dispatch('clienteCreado', $cliente->id);

            // Cerrar el modal
            $this->closeCreateClientModal();

            session()->flash('message', 'Cliente creado exitosamente');
        } catch (\Exception $e) {
            session()->flash('error', 'Error al crear el cliente: ' . $e->getMessage());
        }
    }

    public function searchProducts()
    {
        if (strlen($this->search) >= 2) {
            $query = Producto::where('empresa_id', auth()->user()->empresa_id)
                ->where('status', true)
                ->with(['variants', 'featuredImage']);

            // Buscar por código exacto primero
            $exactMatch = clone $query;
            $productos = $exactMatch->where('code', $this->search)->limit(5)->get();

            // Si no hay coincidencia exacta, buscar por coincidencias parciales
            if ($productos->isEmpty()) {
                $productos = $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%')
                      ->orWhere('code', 'like', '%' . $this->search . '%');
                })
                ->orderBy('name', 'asc')
                ->limit(10)
                ->get();
            }

            $this->searchResults = [];
            foreach ($productos as $producto) {
                $this->searchResults[] = [
                    'id' => $producto->id,
                    'name' => $producto->name,
                    'code' => $producto->code,
                    'description' => $producto->description,
                    'price' => (float) $producto->precio,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'has_variants' => $producto->variants && $producto->variants->count() > 0,
                ];
            }
        } else {
            $this->searchResults = [];
        }
    }

    public function openVariantsModal($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        $this->selectedProductForVariants = $producto;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
        $this->showVariantsModal = true;
    }

    public function closeVariantsModal()
    {
        $this->showVariantsModal = false;
        $this->selectedProductForVariants = null;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
    }

    public function selectVariant($variantId)
    {
        $this->selectedVariant = $variantId;
    }

    public function addVariantToCart()
    {
        if (!$this->selectedProductForVariants || !$this->selectedVariant) {
            session()->flash('error', 'Seleccione una opción del producto');
            return;
        }

        $producto = $this->selectedProductForVariants;

        if ($this->selectedVariant === 'main') {
            // Agregar producto principal
            $this->addProductToCart($producto, $this->variantQuantity);
        } else {
            // Agregar variante
            $variante = $producto->variants()->find($this->selectedVariant);
            if (!$variante) return;

            $precioVariante = $producto->precio + $variante->price_adjustment;
            $values = $variante->formatted_values;

            $existingIndex = collect($this->cart)->search(function ($item) use ($variante) {
                return isset($item['variante_id']) && $item['variante_id'] == $variante->id;
            });

            if ($existingIndex !== false) {
                $this->cart[$existingIndex]['cantidad'] += $this->variantQuantity;
                $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
            } else {
                $this->cart[] = [
                    'producto_id' => $producto->id,
                    'variante_id' => $variante->id,
                    'nombre' => $producto->name . ' - ' . $variante->name . ': ' . $values,
                    'codigo' => $variante->sku ?? $producto->code,
                    'cantidad' => $this->variantQuantity,
                    'precio_unitario' => (float) $precioVariante,
                    'subtotal' => $this->variantQuantity * (float) $precioVariante,
                ];
            }
        }

        $this->closeVariantsModal();
    }

    public function addToCart($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        // Siempre abrir el modal para mostrar el producto principal y sus variantes
        $this->openVariantsModal($productoId);
    }

    private function addProductToCart($producto, $cantidad = 1)
    {
        $existingIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id && !isset($item['variante_id']);
        });

        if ($existingIndex !== false) {
            $this->cart[$existingIndex]['cantidad'] += $cantidad;
            $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
        } else {
            $this->cart[] = [
                'producto_id' => $producto->id,
                'nombre' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $cantidad,
                'precio_unitario' => (float) $producto->precio,
                'subtotal' => $cantidad * (float) $producto->precio,
            ];
        }

        $this->search = '';
        $this->searchResults = [];
    }

    public function removeFromCart($index)
    {
        unset($this->cart[$index]);
        $this->cart = array_values($this->cart);
    }

    public function updateQuantity($index, $quantity)
    {
        if ($quantity < 1) {
            $this->removeFromCart($index);
            return;
        }

        $this->cart[$index]['cantidad'] = $quantity;
        $this->cart[$index]['subtotal'] = $quantity * $this->cart[$index]['precio_unitario'];
    }

    public function getTotal()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function save()
    {
        $this->validate();

        if (empty($this->cart)) {
            session()->flash('error', 'Debe agregar al menos un producto');
            return;
        }

        try {
            DB::beginTransaction();

            $serie = Serie::find($this->serie_id);

            $this->notaVenta->update([
                'total' => $this->getTotal(),
                'cliente_id' => $this->cliente_id,
                'codigo' => $this->codigo,
                'observaciones' => $this->observaciones,
            ]);

            // Eliminar detalles anteriores
            $this->notaVenta->detalles()->delete();

            foreach ($this->cart as $item) {
                $this->notaVenta->detalles()->create([
                    'producto_id' => $item['producto_id'],
                    'producto_variant_id' => $item['variante_id'] ?? null,
                    'cantidad' => $item['cantidad'],
                    'precio_unitario' => $item['precio_unitario'],
                    'subtotal' => $item['subtotal'],
                    'descripcion' => $item['nombre'],
                ]);
            }

            DB::commit();

            session()->flash('message', 'Nota de venta actualizada exitosamente');
            return redirect()->route('admin.notas-venta.show', $this->notaVenta->id);

        } catch (Exception $e) {
            DB::rollback();
            session()->flash('error', 'Error al actualizar la nota de venta: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $this->searchProducts();

        return view('livewire.admin.notas-venta.edit')->layout($this->getLayout());
    }
}
