<?php

namespace App\Livewire\Admin\NotasVenta;

use Livewire\Component;
use App\Models\NotaVenta;
use App\Models\Venta;
use App\Models\DetalleVenta;
use App\Models\Caja;
use App\Models\Serie;
use Illuminate\Support\Facades\DB;

class ConvertToVenta extends Component
{
    public $notaVenta;
    public $confirmingConversion = false;
    public $caja_id = null;
    public $cajas = [];
    public $serie_id = null;
    public $series = [];
    public $metodo_pago = 'efectivo';

    protected $rules = [
        'caja_id' => 'required|exists:cajas,id',
        'serie_id' => 'required|exists:series,id',
        'metodo_pago' => 'required|in:efectivo,transferencia,tarjeta,pago_movil,punto_de_venta',
    ];

    public function mount($notaVentaId)
    {
        $this->notaVenta = NotaVenta::with(['detalles', 'cliente', 'serie'])->findOrFail($notaVentaId);
        $this->metodo_pago = $this->notaVenta->metodo_pago ?? 'efectivo';

        // Cargar cajas activas
        $this->cajas = Caja::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('estado', 'abierta')
            ->get();

        // Cargar series de ventas activas
        $this->series = Serie::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('tipo_documento', 'venta')
            ->where('activo', true)
            ->get();

        // Seleccionar la primera caja y serie por defecto
        if ($this->cajas->count() > 0) {
            $this->caja_id = $this->cajas->first()->id;
        }

        if ($this->series->count() > 0) {
            $this->serie_id = $this->series->first()->id;
        }
    }

    public function confirmarConversion()
    {
        $this->validate();
        $this->confirmingConversion = true;
    }

    public function convertirAVenta()
    {
        // Validar que la nota de venta esté aprobada
        if ($this->notaVenta->estado !== 'aprobada') {
            session()->flash('error', 'Solo se pueden convertir notas de venta aprobadas a ventas.');
            return;
        }

        // Validar que la nota de venta no haya sido convertida ya
        if ($this->notaVenta->venta_id) {
            session()->flash('error', 'Esta nota de venta ya ha sido convertida a venta.');
            return;
        }

        $this->validate();

        DB::beginTransaction();

        try {
            // Obtener la serie seleccionada
            $serie = Serie::findOrFail($this->serie_id);

            // Obtener la siguiente número de venta
            $ultimaVenta = Venta::where('serie_id', $this->serie_id)
                ->orderBy('id', 'desc')
                ->first();

            $siguienteNumero = $ultimaVenta ? ($ultimaVenta->numero_documento ? (int)$ultimaVenta->numero_documento + 1 : 1) : 1;

            // Crear la venta
            $venta = Venta::create([
                'codigo' => 'V-' . time(),
                'total' => $this->notaVenta->total,
                'total_usd' => $this->notaVenta->total / auth()->user()->tasa_cambio_actual ?? 1,
                'total_bs' => $this->notaVenta->total * (auth()->user()->tasa_cambio_actual ?? 1),
                'metodo_pago' => $this->metodo_pago,
                'estado' => 'completada',
                'tipo_documento' => 'venta',
                'numero_documento' => str_pad($siguienteNumero, $serie->longitud_correlativo, '0', STR_PAD_LEFT),
                'serie_id' => $this->serie_id,
                'caja_id' => $this->caja_id,
                'empresa_id' => $this->notaVenta->empresa_id,
                'sucursal_id' => $this->notaVenta->sucursal_id,
                'user_id' => auth()->id(),
                'empleado_id' => $this->notaVenta->user_id,
                'cliente_id' => $this->notaVenta->cliente_id,
                'descuento' => 0,
                'impuesto' => $this->notaVenta->impuestos,
                'observaciones' => "Generado desde nota de venta: {$this->notaVenta->numeroCompleto}",
                'nota_venta_id' => $this->notaVenta->id
            ]);

            // Copiar los detalles de la nota de venta a la venta
            foreach ($this->notaVenta->detalles as $detalle) {
                DetalleVenta::create([
                    'venta_id' => $venta->id,
                    'producto_id' => $detalle->producto_id,
                    'variante_id' => $detalle->producto_variant_id,
                    'descripcion' => $detalle->descripcion,
                    'cantidad' => $detalle->cantidad,
                    'precio_unitario' => $detalle->precio_unitario,
                    'subtotal' => $detalle->subtotal
                ]);
            }

            // Actualizar la nota de venta para indicar que fue convertida
            $this->notaVenta->update([
                'estado' => 'convertida',
                'venta_id' => $venta->id
            ]);

            // Actualizar el correlativo de la serie
            $serie->update([
                'correlativo_actual' => $siguienteNumero
            ]);

            DB::commit();

            session()->flash('message', 'Nota de venta convertida a venta exitosamente.');

            // Redirigir a la venta creada
            return redirect()->route('admin.ventas.show', $venta->id);

        } catch (\Exception $e) {
            DB::rollback();
            session()->flash('error', 'Error al convertir la nota de venta: ' . $e->getMessage());
        }

        $this->confirmingConversion = false;
    }

    public function render()
    {
        return view('livewire.admin.notas-venta.convert-to-venta');
    }
}