<?php

namespace App\Livewire\Admin\NotasCredito;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\NotaCredito;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDynamicLayout;

class Index extends Component
{
    use WithPagination, HasRegionalFormatting, HasDynamicLayout;

    public $search = '';
    public $perPage = 15;
    public $fechaInicio = '';
    public $fechaFin = '';
    public $estado = '';
    public $tipoNota = '';
    public $orderBy = 'created_at';
    public $orderDirection = 'desc';

    protected $paginationTheme = 'bootstrap';
    protected $queryString = [
        'search' => ['except' => ''],
        'fechaInicio' => ['except' => ''],
        'fechaFin' => ['except' => ''],
        'estado' => ['except' => ''],
        'tipoNota' => ['except' => ''],
    ];

    public function mount()
    {
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function limpiarFiltros()
    {
        $this->search = '';
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');
        $this->estado = '';
        $this->tipoNota = '';
        $this->resetPage();
    }

    public function orderBy($field)
    {
        if ($this->orderBy === $field) {
            $this->orderDirection = $this->orderDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->orderBy = $field;
            $this->orderDirection = 'asc';
        }
    }

    public function getTotalNotasProperty()
    {
        return NotaCredito::whereDate('created_at', '>=', $this->fechaInicio)
            ->whereDate('created_at', '<=', $this->fechaFin)
            ->count();
    }

    public function getTotalMontoProperty()
    {
        return NotaCredito::whereDate('created_at', '>=', $this->fechaInicio)
            ->whereDate('created_at', '<=', $this->fechaFin)
            ->sum('total');
    }

    public function render()
    {
        $query = NotaCredito::with(['cliente', 'venta', 'user'])
            ->when($this->search, function ($query) {
                $query->where(function($q) {
                    $q->where('numero', 'like', '%' . $this->search . '%')
                      ->orWhere('serie', 'like', '%' . $this->search . '%')
                      ->orWhere('motivo', 'like', '%' . $this->search . '%')
                      ->orWhereHas('cliente', function ($q) {
                          $q->where('nombres', 'like', '%' . $this->search . '%')
                            ->orWhere('apellidos', 'like', '%' . $this->search . '%');
                      })
                      ->orWhereHas('venta', function ($q) {
                          $q->where('codigo', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->fechaInicio, function ($query) {
                $query->whereDate('fecha_emision', '>=', $this->fechaInicio);
            })
            ->when($this->fechaFin, function ($query) {
                $query->whereDate('fecha_emision', '<=', $this->fechaFin);
            })
            ->when($this->estado, function ($query) {
                $query->where('estado', $this->estado);
            })
            ->when($this->tipoNota, function ($query) {
                $query->where('tipo_nota_credito', $this->tipoNota);
            })
            ->orderBy($this->orderBy, $this->orderDirection);

        $notasCredito = $query->paginate($this->perPage);
        $tiposNotaCredito = NotaCredito::getTiposNotaCredito();
        $estados = ['pendiente' => 'Pendiente', 'procesada' => 'Procesada', 'anulada' => 'Anulada'];

        return view('livewire.admin.notas-credito.index', [
            'notasCredito' => $notasCredito,
            'tiposNotaCredito' => $tiposNotaCredito,
            'estados' => $estados
        ])->layout($this->getLayout());
    }
}