<?php

namespace App\Livewire\Admin\Monitoreo;

use Livewire\Component;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class DatabaseBackup extends Component
{
    use HasDynamicLayout;

    public $selectedTables = [];
    public $allTables = [];
    public $backupDate;
    public $backupTime;
    public $filterFromDate;
    public $filterToDate;
    public $useDataFilter = false;
    public $isProcessing = false;
    
    // Nuevas opciones de respaldo
    public $truncateExistingData = false;
    public $dropTables = false;
    public $dataMode = 'insert'; // 'none', 'delete_insert', 'insert', 'insert_ignore', 'replace'

    public function mount()
    {
        $this->backupDate = now()->format('Y-m-d');
        $this->backupTime = now()->format('H:i');
        $this->filterFromDate = now()->format('Y-m-d');
        $this->filterToDate = now()->format('Y-m-d');
        $this->loadTables();
    }

    public function loadTables()
    {
        $this->allTables = collect(DB::select('SHOW TABLES'))
            ->map(function ($table) {
                $tableName = array_values((array) $table)[0];
                return [
                    'name' => $tableName,
                    'selected' => false,
                    'row_count' => DB::table($tableName)->count()
                ];
            })
            ->toArray();
    }

    public function toggleTable($tableName)
    {
        if (in_array($tableName, $this->selectedTables)) {
            $this->selectedTables = array_filter($this->selectedTables, fn($t) => $t !== $tableName);
        } else {
            $this->selectedTables[] = $tableName;
        }
    }

    public function selectAllTables()
    {
        $this->selectedTables = collect($this->allTables)->pluck('name')->toArray();
    }

    public function deselectAllTables()
    {
        $this->selectedTables = [];
    }

    public function createBackup()
    {
        $this->validate([
            'selectedTables' => 'required|array|min:1',
            'backupDate' => 'required|date',
            'backupTime' => 'required',
            'filterFromDate' => $this->useDataFilter ? 'required|date' : 'nullable',
            'filterToDate' => $this->useDataFilter ? 'required|date|after_or_equal:filterFromDate' : 'nullable'
        ]);

        $this->isProcessing = true;

        try {
            $timestamp = \Carbon\Carbon::parse($this->backupDate . ' ' . $this->backupTime)->format('dmYHi');
            $filename = "respaldo{$timestamp}.sql";
            $backupPath = database_path('Respaldo');
            
            if (!File::exists($backupPath)) {
                File::makeDirectory($backupPath, 0755, true);
            }

            $fullPath = $backupPath . '/' . $filename;
            
            if ($this->useDataFilter) {
                $this->createFilteredBackup($fullPath, $filename);
            } else {
                $this->createFullBackup($fullPath, $filename);
            }

        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al crear el respaldo: ' . $e->getMessage(),
                'timeout' => 8000
            ]);
        } finally {
            $this->isProcessing = false;
        }
    }

    private function createFullBackup($fullPath, $filename)
    {
        // Intentar encontrar mysqldump
        $mysqldumpPath = $this->findMysqldumpPath();
        
        if (!$mysqldumpPath) {
            // Si no se encuentra mysqldump, usar método PHP nativo
            $this->createFilteredBackup($fullPath, $filename);
            return;
        }

        $dbName = config('database.connections.mysql.database');
        $username = config('database.connections.mysql.username');
        $password = config('database.connections.mysql.password');
        $host = config('database.connections.mysql.host');
        $port = config('database.connections.mysql.port', 3306);

        $tables = implode(' ', $this->selectedTables);
        
        // Construir comando con ruta completa y opciones adicionales
        $passwordArg = $password ? "-p{$password}" : '';
        $command = "\"{$mysqldumpPath}\" -h {$host} -P {$port} -u {$username} {$passwordArg} --single-transaction --routines --triggers {$dbName} {$tables} > \"{$fullPath}\"";

        exec($command . ' 2>&1', $output, $returnCode);

        if ($returnCode === 0 && File::exists($fullPath) && filesize($fullPath) > 0) {
            // Agregar comandos de llaves foráneas al archivo generado por mysqldump
            $this->addForeignKeyCommands($fullPath);
            
            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => "Respaldo completo creado: {$filename}",
                'timeout' => 5000
            ]);
        } else {
            // Si falla mysqldump, usar método PHP
            $this->createFilteredBackup($fullPath, $filename);
        }
    }

    private function findMysqldumpPath()
    {
        // Rutas comunes donde puede estar mysqldump
        $possiblePaths = [
            'C:\\laragon\\bin\\mysql\\mysql-8.0.30\\bin\\mysqldump.exe',
            'C:\\laragon\\bin\\mysql\\mysql-5.7.33\\bin\\mysqldump.exe',
            'C:\\xampp\\mysql\\bin\\mysqldump.exe',
            'C:\\Program Files\\MySQL\\MySQL Server 8.0\\bin\\mysqldump.exe',
            'C:\\Program Files\\MySQL\\MySQL Server 5.7\\bin\\mysqldump.exe',
            'mysqldump.exe', // En caso de que esté en PATH
            'mysqldump' // Para sistemas Unix
        ];

        foreach ($possiblePaths as $path) {
            if (file_exists($path) || $this->commandExists($path)) {
                return $path;
            }
        }

        return null;
    }

    private function commandExists($command)
    {
        $whereIsCommand = (PHP_OS == 'WINNT') ? 'where' : 'which';
        $process = proc_open(
            "$whereIsCommand $command",
            [1 => ['pipe', 'w'], 2 => ['pipe', 'w']],
            $pipes
        );
        
        if ($process !== false) {
            $result = stream_get_contents($pipes[1]);
            fclose($pipes[1]);
            fclose($pipes[2]);
            proc_close($process);
            return !empty(trim($result));
        }
        
        return false;
    }

    private function createFilteredBackup($fullPath, $filename)
    {
        $sqlContent = "-- Respaldo filtrado por fechas ({$this->filterFromDate} a {$this->filterToDate})\n";
        $sqlContent .= "-- Generado: " . now()->format('Y-m-d H:i:s') . "\n\n";
        
        // Deshabilitar verificación de llaves foráneas
        $sqlContent .= "-- Deshabilitar verificación de llaves foráneas\n";
        $sqlContent .= "SET FOREIGN_KEY_CHECKS = 0;\n";
        $sqlContent .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n";
        $sqlContent .= "SET AUTOCOMMIT = 0;\n";
        $sqlContent .= "START TRANSACTION;\n\n";
        
        foreach ($this->selectedTables as $table) {
            $sqlContent .= $this->getTableStructure($table);
            $sqlContent .= $this->getFilteredTableData($table);
        }
        
        // Rehabilitar verificación de llaves foráneas
        $sqlContent .= "\n-- Rehabilitar verificación de llaves foráneas\n";
        $sqlContent .= "COMMIT;\n";
        $sqlContent .= "SET FOREIGN_KEY_CHECKS = 1;\n";
        
        File::put($fullPath, $sqlContent);
        
        $this->dispatch('show-toast', [
            'type' => 'success',
            'message' => "Respaldo filtrado creado: {$filename}",
            'timeout' => 5000
        ]);
    }

    private function getTableStructure($table)
    {
        $structure = DB::select("SHOW CREATE TABLE `{$table}`")[0];
        $sql = "\n-- Estructura de tabla `{$table}`\n";
        
        if ($this->dropTables) {
            $sql .= "DROP TABLE IF EXISTS `{$table}`;\n";
        }
        
        $sql .= $structure->{'Create Table'} . ";\n\n";
        
        return $sql;
    }

    private function getFilteredTableData($table)
    {
        if ($this->dataMode === 'none') {
            return "-- Tabla {$table}: Solo estructura (sin datos)\n\n";
        }
        
        $dateColumns = $this->getDateColumns($table);
        
        if (empty($dateColumns)) {
            return "-- Tabla {$table}: Sin columnas de fecha, respaldando todos los datos\n" .
                   $this->getAllTableData($table);
        }
        
        // Filtrar por created_at O updated_at dentro del rango de fechas
        $query = DB::table($table);
        
        if (in_array('created_at', $dateColumns) && in_array('updated_at', $dateColumns)) {
            // Si tiene ambas columnas, filtrar por cualquiera que esté en el rango
            $query->where(function($q) {
                $q->whereBetween('created_at', [$this->filterFromDate . ' 00:00:00', $this->filterToDate . ' 23:59:59'])
                  ->orWhereBetween('updated_at', [$this->filterFromDate . ' 00:00:00', $this->filterToDate . ' 23:59:59']);
            });
        } elseif (in_array('created_at', $dateColumns)) {
            $query->whereBetween('created_at', [$this->filterFromDate . ' 00:00:00', $this->filterToDate . ' 23:59:59']);
        } elseif (in_array('updated_at', $dateColumns)) {
            $query->whereBetween('updated_at', [$this->filterFromDate . ' 00:00:00', $this->filterToDate . ' 23:59:59']);
        } else {
            // Usar la primera columna de fecha encontrada
            $dateColumn = $dateColumns[0];
            $query->whereDate($dateColumn, '>=', $this->filterFromDate)
                  ->whereDate($dateColumn, '<=', $this->filterToDate);
        }
            
        $data = $query->get();
        
        if ($data->isEmpty()) {
            return "-- Tabla {$table}: Sin datos en el rango de fechas\n\n";
        }
        
        return $this->generateInsertStatements($table, $data, 'filtrado');
    }

    private function getDateColumns($table)
    {
        $columns = DB::select("SHOW COLUMNS FROM `{$table}`");
        $dateColumns = [];
        
        foreach ($columns as $column) {
            if (in_array(strtolower($column->Type), ['datetime', 'timestamp', 'date']) || 
                in_array($column->Field, ['created_at', 'updated_at'])) {
                $dateColumns[] = $column->Field;
            }
        }
        
        return $dateColumns;
    }

    private function getAllTableData($table)
    {
        if ($this->dataMode === 'none') {
            return "-- Tabla {$table}: Solo estructura (sin datos)\n\n";
        }
        
        $data = DB::table($table)->get();
        
        if ($data->isEmpty()) {
            return "-- Tabla {$table}: Sin datos\n\n";
        }
        
        return $this->generateInsertStatements($table, $data, 'completo');
    }

    private function generateInsertStatements($table, $data, $type = 'completo')
    {
        if ($data->isEmpty() || $this->dataMode === 'none') {
            return "-- Tabla {$table}: Sin datos\n\n";
        }

        // Obtener las columnas de la tabla
        $columns = $this->getTableColumns($table);
        
        $sql = "-- Datos de tabla `{$table}` ({$type})\n";
        
        // Agregar comando de truncate o delete si es necesario
        if ($this->truncateExistingData) {
            $sql .= "TRUNCATE TABLE `{$table}`;\n";
        } elseif ($this->dataMode === 'delete_insert') {
            $sql .= "DELETE FROM `{$table}`;\n";
        }
        
        // Determinar el tipo de INSERT según el modo
        $insertCommand = $this->getInsertCommand($table, $columns);
        
        $values = [];
        foreach ($data as $row) {
            $rowArray = (array) $row;
            $rowValues = [];
            
            // Asegurar que los valores estén en el mismo orden que las columnas
            foreach ($columns as $column) {
                $value = $rowArray[$column] ?? null;
                $rowValues[] = is_null($value) ? 'NULL' : "'" . addslashes($value) . "'";
            }
            
            $values[] = '(' . implode(', ', $rowValues) . ')';
        }
        
        $sql .= $insertCommand . implode(",\n", $values) . ";\n\n";
        
        return $sql;
    }
    
    private function getInsertCommand($table, $columns)
    {
        $columnsList = "`" . implode('`, `', $columns) . "`";
        
        switch ($this->dataMode) {
            case 'insert_ignore':
                return "INSERT IGNORE INTO `{$table}` ({$columnsList}) VALUES\n";
            case 'replace':
                return "REPLACE INTO `{$table}` ({$columnsList}) VALUES\n";
            case 'delete_insert':
            case 'insert':
            default:
                return "INSERT INTO `{$table}` ({$columnsList}) VALUES\n";
        }
    }

    private function getTableColumns($table)
    {
        $columns = DB::select("SHOW COLUMNS FROM `{$table}`");
        return array_map(function($column) {
            return $column->Field;
        }, $columns);
    }

    private function addForeignKeyCommands($filePath)
    {
        $content = File::get($filePath);
        
        // Agregar comandos al inicio (después de los comentarios iniciales)
        $headerCommands = "\n-- Deshabilitar verificación de llaves foráneas\n";
        $headerCommands .= "SET FOREIGN_KEY_CHECKS = 0;\n";
        $headerCommands .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n";
        $headerCommands .= "SET AUTOCOMMIT = 0;\n";
        $headerCommands .= "START TRANSACTION;\n\n";
        
        // Agregar comandos al final
        $footerCommands = "\n-- Rehabilitar verificación de llaves foráneas\n";
        $footerCommands .= "COMMIT;\n";
        $footerCommands .= "SET FOREIGN_KEY_CHECKS = 1;\n";
        
        // Buscar el final de los comentarios iniciales para insertar los comandos
        $lines = explode("\n", $content);
        $headerInserted = false;
        $newContent = [];
        
        foreach ($lines as $line) {
            $newContent[] = $line;
            
            // Insertar comandos después de los comentarios iniciales de mysqldump
            if (!$headerInserted && (strpos($line, '-- MySQL dump') !== false || strpos($line, '-- Host:') !== false)) {
                // Buscar el final de los comentarios iniciales
                continue;
            }
            
            if (!$headerInserted && trim($line) === '' && !empty($newContent) && strpos(end($newContent), '--') !== false) {
                $newContent[] = $headerCommands;
                $headerInserted = true;
            }
        }
        
        // Agregar comandos al final
        $newContent[] = $footerCommands;
        
        File::put($filePath, implode("\n", $newContent));
    }

    public function render()
    {
        return $this->renderWithLayout('livewire.admin.monitoreo.database-backup', [], [
            'title' => 'Respaldo de Base de Datos',
            'description' => 'Crear respaldos personalizados de la base de datos',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Monitoreo', 'active' => false],
                ['name' => 'Respaldo BD', 'active' => true]
            ]
        ]);
    }
}