<?php

namespace App\Livewire\Admin\Marcas;

use App\Models\Marca;
use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasDynamicLayout;

class Index extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $status = '';
    public $perPage = 10;
    public $sortField = 'nombre';
    public $sortDirection = 'asc';

    // Filtros avanzados
    public $showAdvancedFilters = false;

    // Controles de UI
    public $selectedMarcas = [];

    // Propiedad para las estadísticas
    public $marcaStatsProperty;

    protected $queryString = [
        'search' => ['except' => ''],
        'status' => ['except' => ''],
        'perPage',
        'sortField',
        'sortDirection'
    ];

    protected $listeners = [
        'refreshMarcas' => '$refresh'
    ];

    public function mount()
    {
        $this->loadStats();
    }

    public function updatedSearch()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedStatus()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function loadStats()
    {
        $query = Marca::filterByEmpresaSucursal()
            ->when($this->search, function ($query) {
                $query->where('nombre', 'like', '%' . $this->search . '%');
            })
            ->when($this->status !== '', function ($query) {
                $query->where('activo', $this->status);
            });

        $total = $query->count();
        $activas = $query->where('activo', true)->count();
        $inactivas = $query->where('activo', false)->count();

        $this->marcaStatsProperty = [
            'total' => $total,
            'activas' => $activas,
            'inactivas' => $inactivas
        ];
    }

    public function delete($id)
    {
        $marca = Marca::filterByEmpresaSucursal()->findOrFail($id);

        if ($marca->productos()->count() > 0) {
            $this->dispatch('notify', ['message' => 'No se puede eliminar la marca porque tiene productos asociados.', 'type' => 'error']);
            return;
        }

        $marca->delete();
        $this->dispatch('notify', ['message' => 'Marca eliminada correctamente.', 'type' => 'success']);
        $this->loadStats();
    }

    public function toggleEstado($id)
    {
        $marca = Marca::filterByEmpresaSucursal()->findOrFail($id);
        $marca->activo = !$marca->activo;
        $marca->save();

        $this->loadStats();
        $this->dispatch('notify', ['message' => 'Estado de la marca actualizado correctamente.', 'type' => 'success']);
    }

    public function render()
    {
        $marcas = Marca::filterByEmpresaSucursal()
            ->when($this->search, function ($query) {
                $query->where('nombre', 'like', '%' . $this->search . '%');
            })
            ->when($this->status !== '', function ($query) {
                $query->where('activo', $this->status == '1');
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        return view('livewire.admin.marcas.index', [
            'marcas' => $marcas,
            'stats' => $this->marcaStatsProperty
        ])->layout($this->getLayout(), [
            'title' => 'Lista de Marcas',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Marcas', 'active' => true]
            ]
        ]);
    }
}
