<?php

namespace App\Livewire\Admin\Empleados;

use App\Models\Empleado;
use App\Traits\HasDynamicLayout;
use Livewire\Component;
use Livewire\WithPagination;

class Index extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';
    public $perPage = 10;
    public $statusFilter = '';
    public $departamentoFilter = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'sortBy' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'desc'],
        'statusFilter' => ['except' => ''],
        'departamentoFilter' => ['except' => '']
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function clearFilters()
    {
        $this->search = '';
        $this->statusFilter = '';
        $this->departamentoFilter = '';
        $this->resetPage();
    }

    public function toggleStatus($id)
    {
        $empleado = Empleado::findOrFail($id);
        $empleado->update(['status' => !$empleado->status]);
        
        $status = $empleado->status ? 'activado' : 'desactivado';
        session()->flash('message', "Empleado {$status} exitosamente.");
    }

    public function delete($id)
    {
        $empleado = Empleado::findOrFail($id);
        $empleado->delete();
        
        session()->flash('message', 'Empleado eliminado exitosamente.');
    }

    public function render()
    {
        $empleados = Empleado::filterByEmpresaSucursal()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('nombres', 'like', '%' . $this->search . '%')
                      ->orWhere('apellidos', 'like', '%' . $this->search . '%')
                      ->orWhere('documento', 'like', '%' . $this->search . '%')
                      ->orWhere('email', 'like', '%' . $this->search . '%')
                      ->orWhere('codigo', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter !== '', function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->when($this->departamentoFilter, function ($query) {
                $query->where('departamento', $this->departamentoFilter);
            })
            ->orderBy($this->sortBy, $this->sortDirection)
            ->paginate($this->perPage);

        $stats = [
            'total' => Empleado::filterByEmpresaSucursal()->count(),
            'activos' => Empleado::filterByEmpresaSucursal()->where('status', true)->count(),
            'inactivos' => Empleado::filterByEmpresaSucursal()->where('status', false)->count(),
            'nuevos_mes' => Empleado::filterByEmpresaSucursal()->whereMonth('fecha_ingreso', now()->month)->whereYear('fecha_ingreso', now()->year)->count(),
            'departamentos' => Empleado::filterByEmpresaSucursal()->distinct('departamento')->count('departamento'),
        ];

        $departamentos = Empleado::filterByEmpresaSucursal()
            ->distinct()
            ->pluck('departamento')
            ->filter()
            ->sort();

        return $this->renderWithLayout('livewire.admin.empleados.index', compact('empleados', 'stats', 'departamentos'));
    }
}