<?php

namespace App\Livewire\Admin\Empleados;

use Livewire\Component;
use App\Traits\HasDynamicLayout;
use App\Models\Pedido;
use App\Models\Producto;
use App\Models\Empleado;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class Dashboard extends Component
{
    use HasDynamicLayout;

    public $selectedPeriod = 'week';
    public $showNotifications = true;

    public function getStatsData()
    {
        $empleado = $this->getEmpleado();
        if (!$empleado) return $this->getDefaultStats();

        $empresaId = Auth::user()->empresa_id;
        $today = now()->startOfDay();
        $startOfWeek = now()->startOfWeek();
        $startOfMonth = now()->startOfMonth();

        $weeklyGoal = 25; // Meta más realista
        $ordersThisWeek = $this->getOrdersCount($empleado->id, $startOfWeek, now());
        $weeklyGoalPercentage = $weeklyGoal > 0 ? min(100, round(($ordersThisWeek / $weeklyGoal) * 100)) : 0;

        return [
            'orders_today' => $this->getOrdersCount($empleado->id, $today, now()->endOfDay()),
            'orders_week' => $ordersThisWeek,
            'pending_orders' => $this->getOrdersByStatus($empleado->id, 'Pendiente'),
            'processing_orders' => $this->getOrdersByStatus($empleado->id, 'Procesando'),
            'completed_orders' => $this->getOrdersByStatus($empleado->id, 'entregado'),
            'total_revenue_month' => $this->getRevenueByPeriod($empleado->id, $startOfMonth, now()),
            'avg_order_value' => $this->getAvgOrderValue($empleado->id),
            'weekly_goal' => $weeklyGoal,
            'weekly_progress' => $ordersThisWeek,
            'weekly_goal_percentage' => $weeklyGoalPercentage,
            'efficiency_score' => $this->calculateEfficiencyScore($empleado->id),
        ];
    }

    private function getEmpleado()
    {
        return Empleado::where('user_id', Auth::id())->first();
    }

    private function getDefaultStats()
    {
        return array_fill_keys([
            'orders_today', 'orders_week', 'pending_orders', 'processing_orders',
            'completed_orders', 'total_revenue_month', 'avg_order_value',
            'weekly_goal', 'weekly_progress', 'weekly_goal_percentage', 'efficiency_score'
        ], 0);
    }

    private function getOrdersCount($empleadoId, $start, $end)
    {
        return Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', fn($q) => $q->where('empleadoId', $empleadoId))
            ->whereBetween('created_at', [$start, $end])
            ->count();
    }

    private function getOrdersByStatus($empleadoId, $status)
    {
        return Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', fn($q) => $q->where('empleadoId', $empleadoId))
            ->where('estado', $status)
            ->count();
    }

    private function getRevenueByPeriod($empleadoId, $start, $end)
    {
        return Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', fn($q) => $q->where('empleadoId', $empleadoId))
            ->where('estado', 'entregado')
            ->whereBetween('created_at', [$start, $end])
            ->sum('total_usd');
    }

    private function getAvgOrderValue($empleadoId)
    {
        return Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', fn($q) => $q->where('empleadoId', $empleadoId))
            ->where('estado', 'entregado')
            ->avg('total_usd') ?? 0;
    }

    private function calculateEfficiencyScore($empleadoId)
    {
        $total = $this->getOrdersCount($empleadoId, now()->startOfMonth(), now());
        $completed = $this->getOrdersByStatus($empleadoId, 'entregado');
        return $total > 0 ? round(($completed / $total) * 100) : 0;
    }

    public function getPerformanceData()
    {
        $empleado = $this->getEmpleado();
        if (!$empleado) return ['this_month' => 0, 'last_month' => 0, 'growth_percentage' => 0];

        $thisMonth = $this->getOrdersCount($empleado->id, now()->startOfMonth(), now());
        $lastMonth = $this->getOrdersCount($empleado->id, now()->subMonth()->startOfMonth(), now()->subMonth()->endOfMonth());

        $growth = $lastMonth > 0 ? round((($thisMonth - $lastMonth) / $lastMonth) * 100, 1) : ($thisMonth > 0 ? 100 : 0);

        return [
            'this_month' => $thisMonth,
            'last_month' => $lastMonth,
            'growth_percentage' => $growth
        ];
    }

    public function getRecentOrders()
    {
        $empleado = $this->getEmpleado();
        if (!$empleado) return collect();

        return Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', fn($q) => $q->where('empleadoId', $empleado->id))
            ->with(['user', 'detalles.producto'])
            ->orderBy('created_at', 'desc')
            ->limit(6)
            ->get();
    }

    public function getRecentActivity()
    {
        $userId = Auth::id();
        $empleado = \DB::table('empleados')->where('user_id', $userId)->first();

        if (!$empleado) return collect();

        // Obtener pedidos con sus estados históricos
        $orders = Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', function($query) use ($empleado) {
                $query->where('empleadoId', $empleado->id);
            })
            ->with('historial') // Cargar historial de estados
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get(['id', 'codigo', 'estado', 'created_at', 'updated_at']);

        $activity = collect();

        foreach ($orders as $order) {
            // Agregar evento de creación del pedido
            $activity->push([
                'id' => $order->id,
                'message' => "Nuevo pedido #{$order->codigo} recibido",
                'time' => $order->created_at->diffForHumans(),
                'full_time' => $order->created_at->format('d/m/Y H:i'),
                'icon' => 'ri-shopping-cart-line',
                'type' => 'info'
            ]);

            // Agregar eventos del historial si existen
            if ($order->historial && $order->historial->isNotEmpty()) {
                foreach ($order->historial as $history) {
                    $statusMessage = '';
                    switch ($history->estado) {
                        case 'Pendiente':
                            $statusMessage = 'marcado como pendiente';
                            break;
                        case 'Procesando':
                            $statusMessage = 'marcado como procesando';
                            break;
                        case 'Completado':
                            $statusMessage = 'marcado como completado';
                            break;
                        case 'Cancelado':
                            $statusMessage = 'marcado como cancelado';
                            break;
                        default:
                            $statusMessage = "estado cambiado a {$history->estado}";
                    }

                    $activity->push([
                        'id' => $order->id,
                        'message' => "Pedido #{$order->codigo} {$statusMessage}",
                        'time' => $history->created_at->diffForHumans(),
                        'full_time' => $history->created_at->format('d/m/Y H:i'),
                        'icon' => 'ri-history-line',
                        'type' => 'secondary'
                    ]);
                }
            } else {
                // Si no hay historial, mostrar cambios de estado basados en el estado actual
                if ($order->estado !== 'Pendiente') {
                    $statusMessage = '';
                    switch ($order->estado) {
                        case 'Procesando':
                            $statusMessage = 'marcado como procesando';
                            break;
                        case 'Completado':
                            $statusMessage = 'marcado como completado';
                            break;
                        case 'Cancelado':
                            $statusMessage = 'marcado como cancelado';
                            break;
                        default:
                            $statusMessage = "estado cambiado a {$order->estado}";
                    }

                    $activity->push([
                        'id' => $order->id,
                        'message' => "Pedido #{$order->codigo} {$statusMessage}",
                        'time' => $order->updated_at->diffForHumans(),
                        'full_time' => $order->updated_at->format('d/m/Y H:i'),
                        'icon' => 'ri-history-line',
                        'type' => 'secondary'
                    ]);
                }
            }
        }

        // Ordenar toda la actividad por fecha descendente y limitar a 10 eventos recientes
        return $activity->sortByDesc(function ($item) {
            return \Carbon\Carbon::createFromFormat('d/m/Y H:i', $item['full_time']);
        })->values()->take(10);
    }

    public function showAllActivity()
    {
        // Este método puede ser extendido en el futuro para mostrar todas las actividades
        // Por ahora, simplemente mantenemos la funcionalidad básica
        $this->dispatch('show-all-activity');
    }

    public function updateOrderStatus($orderId, $status)
    {
        $empleado = $this->getEmpleado();
        if (!$empleado) return;

        $pedido = Pedido::find($orderId);
        if ($pedido && $pedido->empleados->contains('id', $empleado->id)) {
            $pedido->update(['estado' => $status]);

            $this->dispatch('alert', [
                'title' => '¡Actualizado!',
                'text' => "Pedido #{$pedido->codigo} actualizado a {$status}",
                'icon' => 'success'
            ]);
        }
    }

    public function viewOrderDetails($orderId)
    {
        return redirect()->route('admin.empleados.pedido.detalle', $orderId);
    }

    public function getQuickActions()
    {
        return [
            ['title' => 'Mis Pedidos', 'icon' => 'ri-shopping-cart-line', 'color' => 'primary', 'action' => 'viewMyOrders'],
            ['title' => 'Pedidos Pendientes', 'icon' => 'ri-time-line', 'color' => 'warning', 'action' => 'viewPendingOrders'],
            ['title' => 'Mi Perfil', 'icon' => 'ri-user-line', 'color' => 'info', 'action' => 'viewProfile'],
            ['title' => 'Actualizar Dashboard', 'icon' => 'ri-refresh-line', 'color' => 'secondary', 'action' => 'refreshDashboard']
        ];
    }

    public function viewMyOrders() { $this->redirect(('/admin/pedidos/historico/' . auth()->id())); }
    public function viewPendingOrders() { $this->redirect(route('admin.pedidos.index', ['estado' => 'Pendiente'])); }
    public function viewProfile() { $this->redirect('/admin/profile'); }
    public function refreshDashboard() { $this->dispatch('$refresh'); }

    public function render()
    {
        return view('livewire.admin.empleados.dashboard', [
            'stats' => $this->getStatsData(),
            'performance' => $this->getPerformanceData(),
            'recentOrders' => $this->getRecentOrders(),
            'recentActivity' => $this->getRecentActivity(),
            'quickActions' => $this->getQuickActions(),
            'empleado' => $this->getEmpleado()
        ])->layout($this->getLayout(), [
            'title' => 'Mi Dashboard',
            'breadcrumb' => [['name' => 'Mi Dashboard', 'active' => true]]
        ]);
    }
}
