<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Traits\HasDynamicLayout;
use App\Models\Pedido;
use App\Models\Cliente;
use App\Models\Venta;
use App\Models\Pago;
use App\Models\Producto;
use App\Models\PaymentSchedule;
use App\Models\User;
use Carbon\Carbon;
use Carbon\CarbonInterval;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Spatie\Permission\Models\Permission;

class Dashboard extends Component
{
    use HasDynamicLayout;

    public $dateRange = 'week';

    protected $queryString = ['dateRange'];

    public function getDateRangeOptions()
    {
        return [
            'week' => 'Últimos 7 días',
            'month' => 'Últimos 30 días',
            'quarter' => 'Trimestre',
            'year' => 'Año'
        ];
    }

    public function mount()
    {
        // Verificar si hay un rango de fechas en la URL
        if (request()->has('dateRange')) {
            $requestedRange = request()->get('dateRange');
            $validRanges = array_keys($this->getDateRangeOptions());

            if (in_array($requestedRange, $validRanges)) {
                $this->dateRange = $requestedRange;
            }
        }
    }

    public function updateDateRange($range)
    {
        $this->dateRange = $range;

        // Emitir evento para actualizar el gráfico en el frontend
        $this->dispatch('updateChart', [
            'chartData' => $this->getSalesChart()->toArray()
        ]);
    }



    public function getStatsData()
    {
        try {
            $empresaId = Auth::user()->empresa_id;
            [$startDate, $endDate] = $this->getDateRange();

            \Log::info('getStatsData - empresaId: ' . $empresaId);
            \Log::info('getStatsData - dateRange: ' . $this->dateRange);
            \Log::info('getStatsData - startDate: ' . $startDate);
            \Log::info('getStatsData - endDate: ' . $endDate);

            $totalOrders = Pedido::where('empresaId', $empresaId)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();

        $pendingOrders = Pedido::where('empresaId', $empresaId)
            ->where('estado', 'pendiente')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        $completedOrders = Pedido::where('empresaId', $empresaId)
             ->where('estado', 'entregado')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        $totalSales = Venta::where('empresa_id', $empresaId)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->sum('total_usd');

        \Log::info('getStatsData - totalSales: ' . $totalSales);

        $totalCustomers = User::where('empresa_id', $empresaId)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        $totalProducts = Producto::where('empresa_id', $empresaId)
            ->count();

        \Log::info('getStatsData - totalOrders: ' . $totalOrders);
        \Log::info('getStatsData - pendingOrders: ' . $pendingOrders);
        \Log::info('getStatsData - completedOrders: ' . $completedOrders);
        \Log::info('getStatsData - totalProducts: ' . $totalProducts);

        // Comparación con período anterior
        $previousRange = $this->getPreviousDateRange();
        $previousOrders = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', $previousRange)
            ->count();

        $previousSales = Venta::where('empresa_id', $empresaId)

            ->whereBetween('created_at', $previousRange)
            ->sum('total_usd');

        $ordersGrowth = $previousOrders > 0 ? (($totalOrders - $previousOrders) / $previousOrders) * 100 : 0;
        $salesGrowth = $previousSales > 0 ? (($totalSales - $previousSales) / $previousSales) * 100 : 0;

        // Productos con stock crítico (5 unidades o menos)
        $criticalStock = Producto::where('empresa_id', $empresaId)
            ->where('quantity', '<=', 5)
            ->where('quantity', '>', 0)
            ->count();

        // Productos sin stock (0 unidades)
        $outOfStock = Producto::where('empresa_id', $empresaId)
            ->where('quantity', 0)
            ->count();

        return [
            'total_orders' => $totalOrders,
            'pending_orders' => $pendingOrders,
            'completed_orders' => $completedOrders,
            'total_sales' => $totalSales,
            'total_customers' => $totalCustomers,
            'total_products' => $totalProducts,
            'orders_growth' => round($ordersGrowth, 1),
            'sales_growth' => round($salesGrowth, 1),
            'critical_stock' => $criticalStock,
            'out_of_stock' => $outOfStock,
            'avg_order_value' => $totalOrders > 0 ? round($totalSales / $totalOrders, 2) : 0,
            'conversion_rate' => $totalOrders > 0 ? round(($completedOrders / $totalOrders) * 100, 1) : 0
        ];
        } catch (\Exception $e) {
            \Log::error('Error en getStatsData: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return [
                'total_orders' => 0,
                'pending_orders' => 0,
                'completed_orders' => 0,
                'total_sales' => 0,
                'total_customers' => 0,
                'total_products' => 0,
                'orders_growth' => 0,
                'sales_growth' => 0,
                'critical_stock' => 0,
                'out_of_stock' => 0,
                'avg_order_value' => 0,
                'conversion_rate' => 0
            ];
        }
    }

    private function getPreviousDateRange()
    {
        $now = now();
        return match($this->dateRange) {
            'week' => [$now->copy()->subDays(13)->startOfDay(), $now->copy()->subDays(7)->endOfDay()],
            'month' => [$now->copy()->subDays(59)->startOfDay(), $now->copy()->subDays(30)->endOfDay()],
            'quarter' => [$now->copy()->subMonths(5)->startOfMonth(), $now->copy()->subMonths(3)->endOfMonth()],
            'year' => [$now->copy()->subMonths(23)->startOfMonth(), $now->copy()->subMonths(12)->endOfMonth()],
            default => [$now->copy()->subDays(13)->startOfDay(), $now->copy()->subDays(7)->endOfDay()]
        };
    }

    public function getRecentOrders()
    {
        return Pedido::where('empresaId', Auth::user()->empresa_id)
            ->with(['user'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
    }

    public function getTopProducts()
    {
        // Get top selling products for the current company
        return DB::table('detalle_ventas')
            ->join('productos', 'detalle_ventas.producto_id', '=', 'productos.id')
            ->where('productos.empresa_id', Auth::user()->empresa_id)
            ->select('productos.name', 'productos.id', DB::raw('SUM(detalle_ventas.cantidad) as total_sold'))
            ->groupBy('productos.id', 'productos.name')
            ->orderBy('total_sold', 'desc')
            ->limit(5)
            ->get();
    }

    public function getSalesChart()
    {
        $empresaId = Auth::user()->empresa_id;
        $now = now();
        $data = [];

        \Log::info('getSalesChart called for empresa: ' . $empresaId . ' with dateRange: ' . $this->dateRange);

        if ($this->dateRange == 'year' || $this->dateRange == 'quarter') {
            $months = $this->dateRange == 'year' ? 12 : 3;

            // Nombres de meses en español
            $meses = ['Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio',
                      'Julio', 'Agosto', 'Septiembre', 'Octubre', 'Noviembre', 'Diciembre'];

            for ($i = $months - 1; $i >= 0; $i--) {
                $monthStart = $now->copy()->subMonths($i)->startOfMonth();
                $monthEnd = $monthStart->copy()->endOfMonth();

                $result = Venta::where('empresa_id', $empresaId)
                    ->whereBetween('created_at', [$monthStart, $monthEnd])
                    ->selectRaw('COUNT(*) as sales')
                    ->first();

                $pedido = Pedido::where('empresaId', $empresaId)
                    ->whereBetween('created_at', [$monthStart, $monthEnd])
                    ->whereIn('estado', ['pagado', 'completado', 'entregado'])
                    ->selectRaw('COUNT(*) as orders')
                    ->first();

                $mesNombre = $meses[$monthStart->month - 1];
                $data[] = [
                    'date' => $monthStart->format('Y-m-01'),
                    'period' => $mesNombre . ' ' . $monthStart->year,
                    'sales' => (float) ($result->sales ?? 0),
                    'orders' => (int) ($pedido->orders ?? 0),
                    'label' => $mesNombre . ' ' . $monthStart->year
                ];
            }
        } else {
            $days = $this->dateRange == 'week' ? 7 : 30;

            // Nombres de días en español
            $dias = ['Dom', 'Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb'];
            // Nombres de meses en español
            $meses = ['Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun',
                      'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'];

            for ($i = $days - 1; $i >= 0; $i--) {
                $date = $now->copy()->subDays($i);

                $result = Venta::where('empresa_id', $empresaId)
                    ->whereDate('created_at', $date)
                    ->selectRaw('COUNT(*) as sales')
                    ->first();

                $pedido = Pedido::where('empresaId', $empresaId)
                    ->whereDate('created_at', $date)
                    ->whereIn('estado', ['pagado', 'completado', 'entregado'])
                    ->selectRaw('COUNT(*) as orders')
                    ->first();

                $periodo = match($this->dateRange) {
                    'week' => $dias[$date->dayOfWeek] . ' ' . $date->day,
                    'month' => $date->day . ' ' . $meses[$date->month - 1],
                    default => $date->day . ' ' . $meses[$date->month - 1]
                };

                $data[] = [
                    'date' => $date->format('Y-m-d'),
                    'period' => $periodo,
                    'sales' => (int) ($result->sales ?? 0),
                    'orders' => (int) ($pedido->orders ?? 0),
                    'label' => $periodo
                ];
            }
        }

        return collect($data);
    }





    public function getAlerts()
    {
        $empresaId = Auth::user()->empresa_id;
        $alerts = collect();

        // Stock bajo
        $lowStock = Producto::where('empresa_id', $empresaId)
            ->where('quantity', '<=', 5)
            ->where('quantity', '>', 0)
            ->count();

        if ($lowStock > 0) {
            $alerts->push([
                'type' => 'warning',
                'icon' => 'ri-alert-line',
                'title' => 'Stock Bajo',
                'message' => "{$lowStock} productos con stock bajo",
                'action' => route('admin.productos.index')
            ]);
        }

        // Ventas pendientes
        $pendingSales = Venta::where('empresa_id', $empresaId)
            ->where('estado', 'pendiente')
            ->where('created_at', '<', now()->subHours(24))
            ->count();

        if ($pendingSales > 0) {
            $alerts->push([
                'type' => 'danger',
                'icon' => 'ri-time-line',
                'title' => 'Ventas Atrasadas',
                'message' => "{$pendingSales} ventas pendientes por más de 24h",
                'action' => route('admin.ventas.index')
            ]);
        }

        return $alerts;
    }

    public function getQuickActions()
    {
        return [
            [
                'title' => 'Nuevo Producto',
                'icon' => 'ri-add-box-line',
                'color' => 'primary',
                'route' => 'admin.productos.create'
            ],
            [
                'title' => 'Ver Ventas',
                'icon' => 'ri-shopping-cart-line',
                'color' => 'success',
                'route' => 'admin.ventas.index'
            ],
            [
                'title' => 'Gestionar Stock',
                'icon' => 'ri-box-3-line',
                'color' => 'warning',
                'route' => 'admin.productos.index'
            ],
            [
                'title' => 'Tasa de Cambio',
                'icon' => 'ri-exchange-dollar-line',
                'color' => 'info',
                'route' => 'admin.exchange-rates',
                'extra' => tasa() ? number_format(tasa()->usd_rate, 2) . ' Bs' : 'N/A'
            ]
        ];
    }

    public function getPerformanceMetrics()
    {
        $empresaId = Auth::user()->empresa_id;
        $startDate = $this->getStartDate();

        // Valor promedio del pedido
        $avgOrderValue = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, now()])
            ->whereIn('estado', ['pagado', 'entregado'])
            ->avg('total_usd') ?? 0;

        // Tasa de conversión (pedidos completados / total pedidos)
        $totalOrders = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, now()])
            ->count();

        $completedOrders = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, now()])
            ->whereIn('estado', ['pagado', 'entregado'])
            ->count();

        $conversionRate = $totalOrders > 0 ? ($completedOrders / $totalOrders) * 100 : 0;

        // Tiempo promedio de entrega (días)
        $avgDeliveryTime = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, now()])
            ->where('estado', 'entregado')
            ->whereNotNull('fecha_entrega')
            ->avg(DB::raw('DATEDIFF(fecha_entrega, created_at)')) ?? 0;

        // Clientes recurrentes (% de clientes con más de 1 pedido)
        $totalCustomers = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, now()])
            ->distinct('userId')
            ->count('userId');

        $repeatCustomers = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, now()])
            ->select('userId', DB::raw('COUNT(*) as total_orders'))
            ->groupBy('userId')
            ->havingRaw('COUNT(*) > 1')
            ->get()
            ->count();

        $repeatCustomersRate = $totalCustomers > 0 ? ($repeatCustomers / $totalCustomers) * 100 : 0;

        // Tasa de cancelación
        $cancelledOrders = Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, now()])
            ->where('estado', 'cancelado')
            ->count();

        $cancellationRate = $totalOrders > 0 ? ($cancelledOrders / $totalOrders) * 100 : 0;

        return [
            'avg_order_value' => round($avgOrderValue, 2),
            'conversion_rate' => round($conversionRate, 1),
            'avg_delivery_time' => round($avgDeliveryTime, 1),
            'repeat_customers' => round($repeatCustomersRate, 1),
            'avg_response_time' => 15, // Valor temporal hasta que se implemente el sistema de soporte
            'cancellation_rate' => round($cancellationRate, 1),
            'inventory_turnover' => 8.5 // Valor temporal hasta que se implemente cálculo real
        ];
    }

    public function getTopCustomers()
    {
        $empresaId = Auth::user()->empresa_id;
        [$startDate, $endDate] = $this->getDateRange();

        return Pedido::where('empresaId', $empresaId)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->whereHas('user', function($query) {
                $query->role('cliente');
            })
            ->with('user')
            ->selectRaw('userId, COUNT(*) as total_orders, SUM(total_usd) as total_spent')
            ->groupBy('userId')
            ->orderBy('total_spent', 'desc')
            ->limit(5)
            ->get()
            ->map(function($pedido) {
                return (object) [
                    'name' => $pedido->user->name ?? 'N/A',
                    'email' => $pedido->user->email ?? 'N/A',
                    'total_orders' => $pedido->total_orders,
                    'total_spent' => $pedido->total_spent
                ];
            });
    }

    public function getTopEmployee()
    {
        $empresaId = Auth::user()->empresa_id;
        [$startDate, $endDate] = $this->getDateRange();

        return Pedido::where('empresaId', $empresaId)
            ->where('estado', 'entregado')
            ->whereBetween('pedidos.created_at', [$startDate, $endDate])
            ->join('empleado_pedido', 'pedidos.id', '=', 'empleado_pedido.pedidoId')
            ->join('empleados', 'empleado_pedido.empleadoId', '=', 'empleados.id')
            ->select('empleados.nombres','empleados.apellidos', 'empleados.id as empleado_id', DB::raw('COUNT(*) as total_deliveries'))
            ->groupBy('empleados.id', 'empleados.nombres', 'empleados.apellidos')
            ->orderBy('total_deliveries', 'desc')
            ->limit(3)
            ->get()
            ->map(function($empleado) {
                return (object) [
                    'name' => $empleado->nombres . ' ' . $empleado->apellidos,
                    'id' => $empleado->empleado_id,
                    'total_deliveries' => $empleado->total_deliveries
                ];
            });
    }

    private function getDateRange()
    {
        $now = now();
        return match($this->dateRange) {
            'week' => [$now->copy()->subDays(6)->startOfDay(), $now->copy()->endOfDay()],
            'month' => [$now->copy()->subDays(29)->startOfDay(), $now->copy()->endOfDay()],
            'quarter' => [$now->copy()->subMonths(2)->startOfMonth(), $now->copy()->endOfMonth()],
            'year' => [$now->copy()->subMonths(11)->startOfMonth(), $now->copy()->endOfMonth()],
            default => [$now->copy()->subDays(6)->startOfDay(), $now->copy()->endOfDay()]
        };
    }

    private function getStartDate()
    {
        return $this->getDateRange()[0];
    }

    private function getEndDate()
    {
        return $this->getDateRange()[1];
    }

    // Método para emitir evento después de actualizar el componente

    public function hydrate()
    {
        //
    }

    public function render()
    {
        $stats = $this->getStatsData();
        $recentOrders = $this->getRecentOrders();
        $topProducts = $this->getTopProducts();
        $salesChart = $this->getSalesChart();
        $alerts = $this->getAlerts();
        $performance = $this->getPerformanceMetrics();
        $topCustomers = $this->getTopCustomers();
        $topEmployees = $this->getTopEmployee();
        $dateRangeOptions = $this->getDateRangeOptions();
        $quickActions = $this->getQuickActions();

        return $this->renderWithLayout('livewire.admin.dashboard-mejorado', [
            'stats' => $stats,
            'recentOrders' => $recentOrders,
            'topProducts' => $topProducts,
            'salesChart' => $salesChart,
            'alerts' => $alerts,
            'performance' => $performance,
            'topCustomers' => $topCustomers,
            'topEmployees' => $topEmployees,
            'dateRangeOptions' => $dateRangeOptions,
            'quickActions' => $quickActions
        ], [
            'title' => 'Dashboard Administrativo',
            'breadcrumb' => [
                'admin.dashboard' => 'Dashboard'
            ]
        ]);
    }
}
