<?php

namespace App\Livewire\Admin\Cupones;

use App\Models\Coupon;
use App\Models\Promotion;
use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasDynamicLayout;
use Carbon\Carbon;

class Index extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $status = '';
    public $promotion_id = '';
    public $perPage = 10;
    public $sortField = 'created_at';
    public $sortDirection = 'desc';

    public $cuponStatsProperty;

    protected $queryString = [
        'search' => ['except' => ''],
        'status' => ['except' => ''],
        'promotion_id' => ['except' => ''],
        'perPage',
        'sortField',
        'sortDirection'
    ];

    protected $listeners = [
        'refreshCupones' => '$refresh'
    ];

    public function mount()
    {
        $this->loadStats();
    }

    public function updatedSearch()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedStatus()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedPromotionId()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function loadStats()
    {
        $empresaId = auth()->user()->empresa_id;
        $now = Carbon::now();

        $total = Coupon::where('empresa_id', $empresaId)->count();
        $activos = Coupon::where('empresa_id', $empresaId)
            ->where('is_active', true)
            ->where(function ($q) use ($now) {
                $q->whereNull('ends_at')->orWhere('ends_at', '>=', $now);
            })
            ->count();
        $usados = Coupon::where('empresa_id', $empresaId)
            ->where('usage_count', '>', 0)
            ->count();
        $expirados = Coupon::where('empresa_id', $empresaId)
            ->whereNotNull('ends_at')
            ->where('ends_at', '<', $now)
            ->count();

        $this->cuponStatsProperty = [
            'total' => $total,
            'activos' => $activos,
            'usados' => $usados,
            'expirados' => $expirados
        ];
    }

    public function delete($id)
    {
        $cupon = Coupon::where('empresa_id', auth()->user()->empresa_id)->findOrFail($id);

        if ($cupon->redemptions()->count() > 0) {
            $this->dispatch('notify', ['message' => 'No se puede eliminar el cupón porque tiene canjes asociados.', 'type' => 'error']);
            return;
        }

        $cupon->delete();
        $this->dispatch('notify', ['message' => 'Cupón eliminado correctamente.', 'type' => 'success']);
        $this->loadStats();
    }

    public function toggleEstado($id)
    {
        $cupon = Coupon::where('empresa_id', auth()->user()->empresa_id)->findOrFail($id);
        $cupon->is_active = !$cupon->is_active;
        $cupon->save();

        $this->loadStats();
        $this->dispatch('notify', ['message' => 'Estado del cupón actualizado correctamente.', 'type' => 'success']);
    }

    public function getStatusLabel($cupon)
    {
        $now = Carbon::now();

        if (!$cupon->is_active) {
            return ['label' => 'Inactivo', 'class' => 'bg-label-secondary'];
        }

        if ($cupon->ends_at && $cupon->ends_at->lt($now)) {
            return ['label' => 'Expirado', 'class' => 'bg-label-danger'];
        }

        if ($cupon->starts_at && $cupon->starts_at->gt($now)) {
            return ['label' => 'Programado', 'class' => 'bg-label-info'];
        }

        if ($cupon->usage_limit && $cupon->usage_count >= $cupon->usage_limit) {
            return ['label' => 'Agotado', 'class' => 'bg-label-warning'];
        }

        return ['label' => 'Activo', 'class' => 'bg-label-success'];
    }

    public function render()
    {
        $empresaId = auth()->user()->empresa_id;
        $now = Carbon::now();

        $cupones = Coupon::where('empresa_id', $empresaId)
            ->with('promotion')
            ->when($this->search, function ($query) {
                $query->where('code', 'like', '%' . $this->search . '%');
            })
            ->when($this->promotion_id !== '', function ($query) {
                $query->where('promotion_id', $this->promotion_id);
            })
            ->when($this->status !== '', function ($query) use ($now) {
                if ($this->status === 'active') {
                    $query->where('is_active', true)
                        ->where(function ($q) use ($now) {
                            $q->whereNull('ends_at')->orWhere('ends_at', '>=', $now);
                        });
                } elseif ($this->status === 'inactive') {
                    $query->where('is_active', false);
                } elseif ($this->status === 'expired') {
                    $query->whereNotNull('ends_at')->where('ends_at', '<', $now);
                }
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $promotions = Promotion::where('empresa_id', $empresaId)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('livewire.admin.cupones.index', [
            'cupones' => $cupones,
            'stats' => $this->cuponStatsProperty,
            'promotions' => $promotions
        ])->layout($this->getLayout(), [
            'title' => 'Lista de Cupones',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Cupones', 'active' => true]
            ]
        ]);
    }
}
