<?php

namespace App\Livewire\Admin\Cupones;

use App\Models\Coupon;
use App\Models\Promotion;
use Livewire\Component;
use App\Traits\HasDynamicLayout;

class Edit extends Component
{
    use HasDynamicLayout;

    public Coupon $cupon;
    public $promotion_id = '';
    public $code = '';
    public $description = '';
    public $usage_limit = null;
    public $usage_limit_per_user = null;
    public $starts_at = null;
    public $ends_at = null;
    public $is_active = true;

    protected function rules()
    {
        return [
            'promotion_id' => 'required|exists:promotions,id',
            'code' => 'required|string|max:50|unique:coupons,code,' . $this->cupon->id,
            'description' => 'nullable|string|max:500',
            'usage_limit' => 'nullable|integer|min:1',
            'usage_limit_per_user' => 'nullable|integer|min:1',
            'starts_at' => 'nullable|date',
            'ends_at' => 'nullable|date|after_or_equal:starts_at',
            'is_active' => 'boolean'
        ];
    }

    protected $messages = [
        'promotion_id.required' => 'Debes seleccionar una promoción.',
        'promotion_id.exists' => 'La promoción seleccionada no es válida.',
        'code.required' => 'El código del cupón es obligatorio.',
        'code.unique' => 'Ya existe un cupón con este código.',
        'code.max' => 'El código no puede exceder 50 caracteres.',
        'description.max' => 'La descripción no puede exceder 500 caracteres.',
        'usage_limit.min' => 'El límite de uso debe ser al menos 1.',
        'usage_limit_per_user.min' => 'El límite por usuario debe ser al menos 1.',
        'ends_at.after_or_equal' => 'La fecha de fin debe ser igual o posterior a la fecha de inicio.'
    ];

    public function mount(Coupon $cupon)
    {
        if ($cupon->empresa_id !== auth()->user()->empresa_id) {
            abort(403);
        }

        $this->cupon = $cupon;
        $this->promotion_id = $cupon->promotion_id;
        $this->code = $cupon->code;
        $this->description = $cupon->description;
        $this->usage_limit = $cupon->usage_limit;
        $this->usage_limit_per_user = $cupon->usage_limit_per_user;
        $this->starts_at = $cupon->starts_at?->format('Y-m-d\TH:i');
        $this->ends_at = $cupon->ends_at?->format('Y-m-d\TH:i');
        $this->is_active = $cupon->is_active;
    }

    public function save()
    {
        $this->validate();

        $promotion = Promotion::where('empresa_id', auth()->user()->empresa_id)
            ->findOrFail($this->promotion_id);

        $this->cupon->update([
            'promotion_id' => $this->promotion_id,
            'code' => strtoupper($this->code),
            'description' => $this->description,
            'usage_limit' => $this->usage_limit,
            'usage_limit_per_user' => $this->usage_limit_per_user,
            'starts_at' => $this->starts_at,
            'ends_at' => $this->ends_at,
            'is_active' => $this->is_active,
        ]);

        $this->dispatch('notify', ['message' => 'Cupón actualizado correctamente.', 'type' => 'success']);
        return redirect()->route('admin.cupones.index');
    }

    public function render()
    {
        $promotions = Promotion::where('empresa_id', auth()->user()->empresa_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        return view('livewire.admin.cupones.edit', [
            'promotions' => $promotions
        ])->layout($this->getLayout(), [
            'title' => 'Editar Cupón',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Cupones', 'route' => 'admin.cupones.index'],
                ['name' => 'Editar', 'active' => true]
            ]
        ]);
    }
}
