<?php

namespace App\Livewire\Admin\Cotizaciones;

use Livewire\Component;
use App\Models\Cotizacion;
use App\Models\DetalleCotizacion;
use App\Models\Cliente;
use App\Models\Producto;
use App\Models\ProductoVariant;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\DB;

class Edit extends Component
{
    use HasRegionalFormatting, HasDynamicLayout;

    public Cotizacion $cotizacion;
    public $fecha_emision;
    public $fecha_vencimiento;
    public $cliente_id;
    public $observaciones;
    public $tasa_impuesto = 0;
    
    public $cart = [];
    public $search = '';
    public $searchResults = [];
    
    // Modal de variantes
    public $showVariantsModal = false;
    public $selectedProductForVariants = null;
    public $selectedVariant = null;
    public $variantQuantity = 1;
    
    public $clientes = [];

    protected $rules = [
        'cliente_id' => 'required|exists:clientes,id',
        'fecha_emision' => 'required|date',
        'fecha_vencimiento' => 'required|date|after_or_equal:fecha_emision',
        'cart' => 'required|array|min:1',
        'cart.*.producto_id' => 'required|exists:productos,id',
        'cart.*.cantidad' => 'required|numeric|min:0.01',
        'cart.*.precio_unitario' => 'required|numeric|min:0.01',
    ];

    public function mount(Cotizacion $cotizacion)
    {
        $this->bootHasRegionalFormatting();
        $this->cotizacion = $cotizacion->load(['detalles.producto', 'detalles.productoVariant']);
        
        $this->fecha_emision = $this->cotizacion->fecha_emision->format('Y-m-d');
        $this->fecha_vencimiento = $this->cotizacion->fecha_vencimiento->format('Y-m-d');
        $this->cliente_id = $this->cotizacion->cliente_id;
        $this->observaciones = $this->cotizacion->observaciones;
        
        // Calcular tasa de impuesto
        if ($this->cotizacion->subtotal > 0) {
            $this->tasa_impuesto = ($this->cotizacion->impuestos / $this->cotizacion->subtotal) * 100;
        }

        // Cargar detalles al carrito
        foreach ($this->cotizacion->detalles as $detalle) {
            $this->cart[] = [
                'id' => $detalle->id,
                'producto_id' => $detalle->producto_id,
                'variante_id' => $detalle->producto_variant_id,
                'descripcion' => $detalle->descripcion,
                'codigo' => $detalle->producto ? $detalle->producto->code : '',
                'cantidad' => $detalle->cantidad,
                'precio_unitario' => $detalle->precio_unitario,
                'subtotal' => $detalle->subtotal
            ];
        }
        
        $this->loadData();
    }

    public function loadData()
    {
        $this->clientes = Cliente::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->orderBy('nombres')
            ->get();
    }

    public function searchProducts()
    {
        if (strlen($this->search) >= 2) {
            $query = Producto::where('empresa_id', auth()->user()->empresa_id)
                ->where('sucursal_id', auth()->user()->sucursal_id)
                ->where('status', true)
                ->with(['variants', 'featuredImage']);

            // Buscar por código exacto primero
            $exactMatch = clone $query;
            $productos = $exactMatch->where('code', $this->search)->limit(5)->get();

            // Si no hay coincidencia exacta, buscar por coincidencias parciales
            if ($productos->isEmpty()) {
                $productos = $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%')
                      ->orWhere('code', 'like', '%' . $this->search . '%');
                })
                ->orderBy('name', 'asc')
                ->limit(10)
                ->get();
            }

            $this->searchResults = [];
            foreach ($productos as $producto) {
                $this->searchResults[] = [
                    'id' => $producto->id,
                    'nombre' => $producto->name,
                    'codigo' => $producto->code,
                    'descripcion' => $producto->description,
                    'precio' => (float) $producto->precio,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'has_variants' => $producto->variants && $producto->variants->count() > 0,
                ];
            }
        } else {
            $this->searchResults = [];
        }
    }

    public function openVariantsModal($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        $this->selectedProductForVariants = $producto;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
        $this->showVariantsModal = true;
    }

    public function closeVariantsModal()
    {
        $this->showVariantsModal = false;
        $this->selectedProductForVariants = null;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
    }

    public function selectVariant($variantId)
    {
        $this->selectedVariant = $variantId;
    }

    public function addVariantToCart()
    {
        if (!$this->selectedProductForVariants || !$this->selectedVariant) {
            session()->flash('error', 'Seleccione una opción del producto');
            return;
        }

        $producto = $this->selectedProductForVariants;

        if ($this->selectedVariant === 'main') {
            // Agregar producto principal
            $this->addProductToCart($producto, $this->variantQuantity);
        } else {
            // Agregar variante
            $variante = $producto->variants()->find($this->selectedVariant);
            if (!$variante) return;

            $precioVariante = $producto->precio + $variante->price_adjustment;
            $values = $variante->formatted_values;
            
            $existingIndex = collect($this->cart)->search(function ($item) use ($variante) {
                return isset($item['variante_id']) && $item['variante_id'] == $variante->id;
            });

            if ($existingIndex !== false) {
                $this->cart[$existingIndex]['cantidad'] += $this->variantQuantity;
                $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
            } else {
                $this->cart[] = [
                    'id' => null, // Nuevo item
                    'producto_id' => $producto->id,
                    'variante_id' => $variante->id,
                    'descripcion' => $producto->name . ' - ' . $variante->name . ': ' . $values,
                    'codigo' => $variante->sku ?? $producto->code,
                    'cantidad' => $this->variantQuantity,
                    'precio_unitario' => (float) $precioVariante,
                    'subtotal' => $this->variantQuantity * (float) $precioVariante,
                ];
            }
        }

        $this->closeVariantsModal();
    }

    public function addToCart($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        // Siempre abrir el modal para mostrar el producto principal y sus variantes
        $this->openVariantsModal($productoId);
    }

    private function addProductToCart($producto, $cantidad = 1)
    {
        $existingIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id && !isset($item['variante_id']);
        });

        if ($existingIndex !== false) {
            $this->cart[$existingIndex]['cantidad'] += $cantidad;
            $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
        } else {
            $this->cart[] = [
                'id' => null, // Nuevo item
                'producto_id' => $producto->id,
                'descripcion' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $cantidad,
                'precio_unitario' => (float) $producto->precio,
                'subtotal' => $cantidad * (float) $producto->precio,
            ];
        }

        $this->search = '';
        $this->searchResults = [];
    }

    public function removeFromCart($index)
    {
        unset($this->cart[$index]);
        $this->cart = array_values($this->cart);
    }

    public function updateQuantity($index, $quantity)
    {
        if ($quantity < 0.01) {
            $this->removeFromCart($index);
            return;
        }

        $this->cart[$index]['cantidad'] = $quantity;
        $this->cart[$index]['subtotal'] = $quantity * $this->cart[$index]['precio_unitario'];
    }

    public function getSubtotal()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function getImpuestos()
    {
        return $this->getSubtotal() * ($this->tasa_impuesto / 100);
    }

    public function getTotal()
    {
        return $this->getSubtotal() + $this->getImpuestos();
    }

    public function updatedTasaImpuesto()
    {
        // Trigger recalculation
    }

    public function save()
    {
        if ($this->cotizacion->estado !== 'pendiente') {
            session()->flash('error', 'Solo se pueden editar cotizaciones pendientes');
            return;
        }

        $this->validate();

        if (empty($this->cart)) {
            session()->flash('error', 'Debe agregar al menos un producto');
            return;
        }

        try {
            DB::transaction(function () {
                // Actualizar cotización
                $this->cotizacion->update([
                    'fecha_emision' => $this->fecha_emision,
                    'fecha_vencimiento' => $this->fecha_vencimiento,
                    'cliente_id' => $this->cliente_id,
                    'observaciones' => $this->observaciones,
                    'subtotal' => $this->getSubtotal(),
                    'impuestos' => $this->getImpuestos(),
                    'total' => $this->getTotal()
                ]);

                // Eliminar detalles existentes
                $this->cotizacion->detalles()->delete();

                // Crear nuevos detalles
                foreach ($this->cart as $item) {
                    DetalleCotizacion::create([
                        'cotizacion_id' => $this->cotizacion->id,
                        'producto_id' => $item['producto_id'],
                        'producto_variant_id' => $item['variante_id'] ?? null,
                        'descripcion' => $item['descripcion'],
                        'cantidad' => $item['cantidad'],
                        'precio_unitario' => $item['precio_unitario'],
                        'subtotal' => $item['subtotal']
                    ]);
                }
            });

            session()->flash('message', 'Cotización actualizada exitosamente');
            return redirect()->route('admin.cotizaciones.show', $this->cotizacion->id);

        } catch (\Exception $e) {
            session()->flash('error', 'Error al actualizar la cotización: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $this->searchProducts();
        
        return view('livewire.admin.cotizaciones.edit')->layout($this->getLayout());
    }
}