<?php

namespace App\Livewire\Admin\Cotizaciones;

use Livewire\Component;
use App\Models\Cotizacion;
use App\Models\DetalleCotizacion;
use App\Models\Cliente;
use App\Models\Serie;
use App\Models\Producto;
use App\Models\ProductoVariant;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\DB;

class Create extends Component
{
    use HasRegionalFormatting, HasDynamicLayout;

    public $serie = 'COT01';
    public $numero;
    public $fecha_emision;
    public $fecha_vencimiento;
    public $cliente_id;
    public $observaciones = '';
    public $tasa_impuesto = 0;
    
    public $cart = [];
    public $search = '';
    public $searchResults = [];
    
    // Modal de variantes
    public $showVariantsModal = false;
    public $selectedProductForVariants = null;
    public $selectedVariant = null;
    public $variantQuantity = 1;
    
    public $clientes = [];
    public $productos = [];
    
    // Modal de cliente
    public $showClientModal = false;
    public $clienteNombres = '';
    public $clienteApellidos = '';
    public $clienteEmail = '';
    public $clienteTelefono = '';
    public $clienteDireccion = '';

    protected $rules = [
        'cliente_id' => 'required|exists:clientes,id',
        'fecha_emision' => 'required|date',
        'fecha_vencimiento' => 'required|date|after_or_equal:fecha_emision',
        'cart' => 'required|array|min:1',
        'cart.*.producto_id' => 'required|exists:productos,id',
        'cart.*.cantidad' => 'required|numeric|min:0.01',
        'cart.*.precio_unitario' => 'required|numeric|min:0.01',
    ];

    public function mount()
    {
        $this->bootHasRegionalFormatting();
        $this->fecha_emision = now()->format('Y-m-d');
        $this->fecha_vencimiento = now()->addDays(30)->format('Y-m-d');
        $this->generateNumber();
        $this->loadData();
    }

    public function loadData()
    {
        $this->clientes = Cliente::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->orderBy('nombres')
            ->get();
    }

    public function abrirModalCliente()
    {
        $this->showClientModal = true;
        $this->resetClientModal();
    }

    public function cerrarModalCliente()
    {
        $this->showClientModal = false;
        $this->resetClientModal();
    }

    public function resetClientModal()
    {
        $this->clienteNombres = '';
        $this->clienteApellidos = '';
        $this->clienteEmail = '';
        $this->clienteTelefono = '';
        $this->clienteDireccion = '';
    }

    public function guardarCliente()
    {
        $this->validate([
            'clienteNombres' => 'required|string|max:255',
            'clienteApellidos' => 'required|string|max:255',
            'clienteEmail' => 'nullable|email|unique:clientes,email',
            'clienteTelefono' => 'nullable|string|max:20',
        ]);

        try {
            $cliente = Cliente::create([
                'nombres' => $this->clienteNombres,
                'apellidos' => $this->clienteApellidos,
                'email' => $this->clienteEmail,
                'telefono' => $this->clienteTelefono,
                'direccion' => $this->clienteDireccion,
                'empresa_id' => auth()->user()->empresa_id,
                'sucursal_id' => auth()->user()->sucursal_id,
            ]);

            $this->cliente_id = $cliente->id;
            $this->loadData();
            $this->cerrarModalCliente();
            
            $this->dispatch('show-message', [
                'type' => 'success',
                'message' => 'Cliente registrado exitosamente'
            ]);
        } catch (\Exception $e) {
            $this->dispatch('show-message', [
                'type' => 'error',
                'message' => 'Error al registrar cliente: ' . $e->getMessage()
            ]);
        }
    }

    public function generateNumber()
    {
        $serie = Serie::where('tipo_documento', 'cotizacion')
            ->where('serie', $this->serie)
            ->where('activo', true)
            ->first();

        if ($serie) {
            // Buscar el próximo número disponible
            $nextNumber = $serie->correlativo_actual + 1;
            
            while (Cotizacion::where('serie', $this->serie)
                ->where('numero', str_pad($nextNumber, 8, '0', STR_PAD_LEFT))
                ->where('empresa_id', auth()->user()->empresa_id)
                ->exists()) {
                $nextNumber++;
            }
            
            $this->numero = str_pad($nextNumber, 8, '0', STR_PAD_LEFT);
        }
    }

    public function searchProducts()
    {
        if (strlen($this->search) >= 2) {
            $query = Producto::where('empresa_id', auth()->user()->empresa_id)
                ->where('sucursal_id', auth()->user()->sucursal_id)
                ->where('status', true)
                ->with(['variants', 'featuredImage']);

            // Buscar por código exacto primero
            $exactMatch = clone $query;
            $productos = $exactMatch->where('code', $this->search)->limit(5)->get();

            // Si no hay coincidencia exacta, buscar por coincidencias parciales
            if ($productos->isEmpty()) {
                $productos = $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%')
                      ->orWhere('code', 'like', '%' . $this->search . '%');
                })
                ->orderBy('name', 'asc')
                ->limit(10)
                ->get();
            }

            $this->searchResults = [];
            foreach ($productos as $producto) {
                $this->searchResults[] = [
                    'id' => $producto->id,
                    'nombre' => $producto->name,
                    'codigo' => $producto->code,
                    'descripcion' => $producto->description,
                    'precio' => (float) $producto->precio,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'has_variants' => $producto->variants && $producto->variants->count() > 0,
                ];
            }
        } else {
            $this->searchResults = [];
        }
    }

    public function openVariantsModal($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        $this->selectedProductForVariants = $producto;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
        $this->showVariantsModal = true;
    }

    public function closeVariantsModal()
    {
        $this->showVariantsModal = false;
        $this->selectedProductForVariants = null;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
    }

    public function selectVariant($variantId)
    {
        $this->selectedVariant = $variantId;
    }

    public function addVariantToCart()
    {
        if (!$this->selectedProductForVariants || !$this->selectedVariant) {
            session()->flash('error', 'Seleccione una opción del producto');
            return;
        }

        $producto = $this->selectedProductForVariants;

        if ($this->selectedVariant === 'main') {
            // Agregar producto principal
            $this->addProductToCart($producto, $this->variantQuantity);
        } else {
            // Agregar variante
            $variante = $producto->variants()->find($this->selectedVariant);
            if (!$variante) return;

            $precioVariante = $producto->precio + $variante->price_adjustment;
            $values = $variante->formatted_values;
            
            $existingIndex = collect($this->cart)->search(function ($item) use ($variante) {
                return isset($item['variante_id']) && $item['variante_id'] == $variante->id;
            });

            if ($existingIndex !== false) {
                $this->cart[$existingIndex]['cantidad'] += $this->variantQuantity;
                $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
            } else {
                $this->cart[] = [
                    'producto_id' => $producto->id,
                    'variante_id' => $variante->id,
                    'descripcion' => $producto->name . ' - ' . $variante->name . ': ' . $values,
                    'codigo' => $variante->sku ?? $producto->code,
                    'cantidad' => $this->variantQuantity,
                    'precio_unitario' => (float) $precioVariante,
                    'subtotal' => $this->variantQuantity * (float) $precioVariante,
                ];
            }
        }

        $this->closeVariantsModal();
    }

    public function addToCart($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        // Siempre abrir el modal para mostrar el producto principal y sus variantes
        $this->openVariantsModal($productoId);
    }

    private function addProductToCart($producto, $cantidad = 1)
    {
        $existingIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id && !isset($item['variante_id']);
        });

        if ($existingIndex !== false) {
            $this->cart[$existingIndex]['cantidad'] += $cantidad;
            $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
        } else {
            $this->cart[] = [
                'producto_id' => $producto->id,
                'descripcion' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $cantidad,
                'precio_unitario' => (float) $producto->precio,
                'subtotal' => $cantidad * (float) $producto->precio,
            ];
        }

        $this->search = '';
        $this->searchResults = [];
    }

    public function removeFromCart($index)
    {
        unset($this->cart[$index]);
        $this->cart = array_values($this->cart);
    }

    public function updateQuantity($index, $quantity)
    {
        if ($quantity < 0.01) {
            $this->removeFromCart($index);
            return;
        }

        $this->cart[$index]['cantidad'] = $quantity;
        $this->cart[$index]['subtotal'] = $quantity * $this->cart[$index]['precio_unitario'];
    }

    public function getSubtotal()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function getImpuestos()
    {
        return $this->getSubtotal() * ($this->tasa_impuesto / 100);
    }

    public function getTotal()
    {
        return $this->getSubtotal() + $this->getImpuestos();
    }

    public function updatedTasaImpuesto()
    {
        // Trigger recalculation
    }

    public function save()
    {
        $this->validate();

        if (empty($this->cart)) {
            session()->flash('error', 'Debe agregar al menos un producto');
            return;
        }

        try {
            DB::transaction(function () {
                $cotizacion = Cotizacion::create([
                    'serie' => $this->serie,
                    'numero' => $this->numero,
                    'fecha_emision' => $this->fecha_emision,
                    'fecha_vencimiento' => $this->fecha_vencimiento,
                    'cliente_id' => $this->cliente_id,
                    'observaciones' => $this->observaciones,
                    'subtotal' => $this->getSubtotal(),
                    'impuestos' => $this->getImpuestos(),
                    'total' => $this->getTotal(),
                    'estado' => 'pendiente',
                    'empresa_id' => auth()->user()->empresa_id,
                    'sucursal_id' => auth()->user()->sucursal_id,
                    'user_id' => auth()->id()
                ]);

                foreach ($this->cart as $item) {
                    DetalleCotizacion::create([
                        'cotizacion_id' => $cotizacion->id,
                        'producto_id' => $item['producto_id'],
                        'producto_variant_id' => $item['variante_id'] ?? null,
                        'descripcion' => $item['descripcion'],
                        'cantidad' => $item['cantidad'],
                        'precio_unitario' => $item['precio_unitario'],
                        'subtotal' => $item['subtotal']
                    ]);
                }

                // Actualizar serie con el número usado
                $serie = Serie::where('tipo_documento', 'cotizacion')
                    ->where('serie', $this->serie)
                    ->where('activo', true)
                    ->first();
                if ($serie) {
                    $numeroUsado = (int) $this->numero;
                    if ($numeroUsado > $serie->correlativo_actual) {
                        $serie->update(['correlativo_actual' => $numeroUsado]);
                    }
                }
            });

            session()->flash('message', 'Cotización creada exitosamente');
            return redirect()->route('admin.cotizaciones.show', $cotizacion->id);

        } catch (\Exception $e) {
            session()->flash('error', 'Error al crear la cotización: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $this->searchProducts();
        
        return view('livewire.admin.cotizaciones.create')->layout($this->getLayout());
    }
}
