<?php

namespace App\Livewire\Admin\Categorias;

use App\Models\Categoria;
use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasDynamicLayout;

class Index extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $status = '';
    public $perPage = 10;
    public $sortField = 'nombre';
    public $sortDirection = 'asc';

    // Filtros avanzados
    public $showAdvancedFilters = false;

    // Controles de UI
    public $selectedCategorias = [];

    // Propiedad para las estadísticas
    public $categoriaStatsProperty;

    protected $queryString = [
        'search' => ['except' => ''],
        'status' => ['except' => ''],
        'perPage',
        'sortField',
        'sortDirection'
    ];

    protected $listeners = [
        'refreshCategorias' => '$refresh'
    ];

    public function mount()
    {
        $this->loadStats();
    }

    public function updatedSearch()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedStatus()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
        $this->loadStats();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function loadStats()
    {
        $query = Categoria::filterByEmpresaSucursal()
            ->when($this->search, function ($query) {
                $query->where('nombre', 'like', '%' . $this->search . '%');
            })
            ->when($this->status !== '', function ($query) {
                $query->where('activo', $this->status);
            });

        $total = $query->count();
        $activas = $query->where('activo', true)->count();
        $inactivas = $query->where('activo', false)->count();

        $this->categoriaStatsProperty = [
            'total' => $total,
            'activas' => $activas,
            'inactivas' => $inactivas
        ];
    }

    public function delete($id)
    {
        $categoria = Categoria::filterByEmpresaSucursal()->findOrFail($id);

        if ($categoria->productos()->count() > 0) {
            $this->dispatch('notify', ['message' => 'No se puede eliminar la categoría porque tiene productos asociados.', 'type' => 'error']);
            return;
        }

        $categoria->delete();
        $this->dispatch('notify', ['message' => 'Categoría eliminada correctamente.', 'type' => 'success']);
        $this->loadStats();
    }

    public function toggleEstado($id)
    {
        $categoria = Categoria::filterByEmpresaSucursal()->findOrFail($id);
        $categoria->activo = !$categoria->activo;
        $categoria->save();

        $this->loadStats();
        $this->dispatch('notify', ['message' => 'Estado de la categoría actualizado correctamente.', 'type' => 'success']);
    }

    public function render()
    {
        $categorias = Categoria::filterByEmpresaSucursal()
            ->when($this->search, function ($query) {
                $query->where('nombre', 'like', '%' . $this->search . '%');
            })
            ->when($this->status !== '', function ($query) {
                $query->where('activo', $this->status == '1');
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

          return $this->renderWithLayout('livewire.admin.categorias.index',[
             'categorias' => $categorias,
             'stats' => $this->categoriaStatsProperty
          ], [
            'description' => 'Gestión de categorias',
        ]);
    }
}
