<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use App\Traits\HasDynamicLayout;
use Spatie\Activitylog\Models\Activity;
use Livewire\WithPagination;
use App\Models\User;
use Carbon\Carbon;

class ActivityLog extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $userFilter = '';
    public $dateRange = '';
    public $actionFilter = '';
    public $subjectFilter = '';
    public $perPage = 15;
    public $showStats = true;

    protected $queryString = ['search', 'userFilter', 'dateRange', 'actionFilter', 'subjectFilter'];

    public function render()
    {
        $activities = $this->getActivities();
        $users = User::select('id', 'name')->orderBy('name')->get();
        $actions = Activity::select('description')->distinct()->pluck('description');
        $subjects = Activity::select('subject_type')->distinct()->whereNotNull('subject_type')->pluck('subject_type');
        $stats = $this->showStats ? $this->getStats() : null;

        return $this->renderWithLayout('livewire.admin.activity-log', [
            'activities' => $activities,
            'users' => $users,
            'actions' => $actions,
            'subjects' => $subjects,
            'stats' => $stats
        ], [
            'title' => 'Registro de Actividades',
            'breadcrumb' => [
                'admin.dashboard' => 'Dashboard',
                'admin.activity-log' => 'Actividades'
            ]
        ]);
    }

    private function getActivities()
    {
        return Activity::with(['causer:id,name,email', 'subject'])
            ->when($this->search, function ($query) {
                $query->where(function($q) {
                    $q->where('description', 'like', '%' . $this->search . '%')
                      ->orWhere('subject_type', 'like', '%' . $this->search . '%')
                      ->orWhereHas('causer', function ($subQ) {
                          $subQ->where('name', 'like', '%' . $this->search . '%')
                               ->orWhere('email', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->userFilter, fn($query) => $query->where('causer_id', $this->userFilter))
            ->when($this->actionFilter, fn($query) => $query->where('description', $this->actionFilter))
            ->when($this->subjectFilter, fn($query) => $query->where('subject_type', $this->subjectFilter))
            ->when($this->dateRange, function ($query) {
                $dates = $this->getDateRange();
                if ($dates) $query->whereBetween('created_at', $dates);
            })
            ->orderBy('created_at', 'desc')
            ->paginate($this->perPage);
    }

    private function getDateRange()
    {
        return match($this->dateRange) {
            'today' => [now()->startOfDay(), now()->endOfDay()],
            'yesterday' => [now()->subDay()->startOfDay(), now()->subDay()->endOfDay()],
            'week' => [now()->startOfWeek(), now()->endOfWeek()],
            'month' => [now()->startOfMonth(), now()->endOfMonth()],
            'last_week' => [now()->subWeek()->startOfWeek(), now()->subWeek()->endOfWeek()],
            'last_month' => [now()->subMonth()->startOfMonth(), now()->subMonth()->endOfMonth()],
            default => null,
        };
    }

    private function getStats()
    {
        $dateRange = $this->getDateRange() ?? [now()->startOfMonth(), now()->endOfMonth()];

        return [
            'total' => Activity::whereBetween('created_at', $dateRange)->count(),
            'by_action' => Activity::whereBetween('created_at', $dateRange)
                ->selectRaw('description, COUNT(*) as count')
                ->groupBy('description')
                ->pluck('count', 'description'),
            'by_user' => Activity::whereBetween('created_at', $dateRange)
                ->with('causer:id,name')
                ->get()
                ->groupBy('causer.name')
                ->map->count()
                ->take(5),
            'recent_peak' => Activity::whereBetween('created_at', [now()->subHours(24), now()])
                ->selectRaw('DATE_FORMAT(created_at, "%H:00") as hour, COUNT(*) as count')
                ->groupBy('hour')
                ->pluck('count', 'hour')
        ];
    }

    public function clearFilters()
    {
        $this->reset(['search', 'userFilter', 'dateRange', 'actionFilter', 'subjectFilter']);
        $this->resetPage();
    }

    public function toggleStats()
    {
        $this->showStats = !$this->showStats;
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function getActionColor($action)
    {
        return match($action) {
            'created' => 'success',
            'updated' => 'warning',
            'deleted' => 'danger',
            'login' => 'info',
            'logout' => 'secondary',
            'restored' => 'primary',
            default => 'dark',
        };
    }

    public function getActionIcon($action)
    {
        return match($action) {
            'created' => 'ri-add-circle-line',
            'updated' => 'ri-edit-line',
            'deleted' => 'ri-delete-bin-line',
            'login' => 'ri-login-circle-line',
            'logout' => 'ri-logout-circle-line',
            'restored' => 'ri-refresh-line',
            default => 'ri-information-line',
        };
    }

    public function export()
    {
        $activities = Activity::with(['causer', 'subject'])
            ->when($this->search, function ($query) {
                $query->where('description', 'like', '%' . $this->search . '%')
                    ->orWhereHas('causer', function ($q) {
                        $q->where('name', 'like', '%' . $this->search . '%');
                    });
            })
            ->when($this->userFilter, function ($query) {
                $query->where('causer_id', $this->userFilter);
            })
            ->when($this->dateRange, function ($query) {
                $dates = match($this->dateRange) {
                    'today' => [now()->startOfDay(), now()->endOfDay()],
                    'yesterday' => [now()->subDay()->startOfDay(), now()->subDay()->endOfDay()],
                    'week' => [now()->startOfWeek(), now()->endOfWeek()],
                    'month' => [now()->startOfMonth(), now()->endOfMonth()],
                    default => null,
                };

                if ($dates) {
                    $query->whereBetween('created_at', $dates);
                }
            })
            ->orderBy('created_at', 'desc')
            ->get();

        $headers = [
            'Fecha y Hora',
            'Usuario',
            'Acción',
            'Elemento',
            'ID del Elemento',
            'Detalles'
        ];

        $data = $activities->map(function ($activity) {
            return [
                'created_at' => $activity->created_at->format('d/m/Y H:i:s'),
                'user' => $activity->causer ? $activity->causer->name : 'Sistema',
                'action' => $activity->description,
                'subject_type' => $activity->subject ? class_basename($activity->subject_type) : 'N/A',
                'subject_id' => $activity->subject_id,
                'properties' => json_encode($activity->properties, JSON_UNESCAPED_UNICODE)
            ];
        });

        $filename = 'registro_actividades_' . now()->format('Y-m-d_His') . '.xlsx';

        return response()->streamDownload(function () use ($headers, $data) {
            $file = fopen('php://output', 'w');

            // Escribir encabezados
            fputcsv($file, $headers);

            // Escribir datos
            foreach ($data as $row) {
                fputcsv($file, $row);
            }

            fclose($file);
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }
}




