<?php

namespace App\Jobs;

use App\Models\User;
use App\Services\WhatsAppService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;

class SendWelcomeWhatsAppMessage implements ShouldQueue
{
    use Queueable;

    protected $user;
    protected $credentialToken;
    
    public $tries = 3;
    public $backoff = [10, 30, 60]; // Reintentos en 10s, 30s, 60s

    /**
     * Create a new job instance.
     */
    public function __construct(User $user, $credentialToken)
    {
        $this->user = $user;
        $this->credentialToken = $credentialToken;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $whatsappService = new WhatsAppService();
            
            // Obtener nombre de la empresa
            $empresa = \App\Models\Empresa::find(1);
            $empresaNombre = $empresa ? $empresa->name : 'nuestra tienda';
            
            $message = "🎉 *¡Bienvenido a {$empresaNombre}!* 🎉\n\n";
            $message .= "*Hola {$this->user->name}!*\n\n";
            $message .= "Gracias por registrarte en nuestra plataforma. Ahora puedes:\n\n";
            $message .= "🛒 Realizar pedidos fácilmente\n";
            $message .= "📱 Recibir notificaciones de tus pedidos\n";
            $message .= "⭐ Guardar productos en favoritos\n";
            $message .= "🚚 Seguimiento de entregas\n\n";
            // Obtener credenciales del token temporal
            $credentials = \App\Services\TemporaryTokenService::getCredentials($this->credentialToken);
            
            if ($credentials) {
                $message .= "🔑 *Datos de acceso:*\n";
                $message .= "Usuario: {$credentials['username']}\n";
                $message .= "Contraseña: {$credentials['password']}\n\n";
            } else {
                $message .= "🔑 *Datos de acceso:*\n";
                $message .= "Usuario: {$this->user->username}\n";
                $message .= "Contraseña: [Contacta soporte para recuperar]\n\n";
            }
            $message .= "¡Esperamos que disfrutes de tu experiencia de compra!\n\n";
            $message .= "Si tienes alguna pregunta, no dudes en contactarnos.";
            
            // Formatear el número de teléfono antes de enviar
            $cleanPhone = $whatsappService->cleanPhoneNumber($this->user->telefono);
            $whatsappService->sendMessage($cleanPhone, $message);
            
            Log::info('Mensaje de bienvenida enviado exitosamente', ['user_id' => $this->user->id]);
            
        } catch (\Exception $e) {
            Log::error('Error enviando mensaje de bienvenida: ' . $e->getMessage(), [
                'user_id' => $this->user->id,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts()
            ]);
            
            // Si es el último intento, registrar fallo definitivo
            if ($this->attempts() >= $this->tries) {
                Log::error('Fallo definitivo enviando mensaje de bienvenida después de ' . $this->tries . ' intentos', [
                    'user_id' => $this->user->id
                ]);
            }
            
            throw $e; // Re-lanzar para activar reintentos
        }
    }
}