<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class GeocodingController extends Controller
{
    /**
     * Método reverse para geocodificación inversa (de coordenadas a dirección).
     */
    public function reverse(Request $request)
    {
        $lat = $request->query('lat');
        $lon = $request->query('lon');

        if (!$lat || !$lon) {
            return response()->json(['error' => 'Latitude and longitude are required'], 400);
        }

        // Validar coordenadas
        if (!is_numeric($lat) || !is_numeric($lon) ||
            $lat < -90 || $lat > 90 ||
            $lon < -180 || $lon > 180) {
            return response()->json(['error' => 'Invalid coordinates'], 400);
        }

        try {
            $cacheKey = "reverse_geocode_{$lat}_{$lon}";

            // Verificar en caché primero
            if (Cache::has($cacheKey)) {
                return response()->json(Cache::get($cacheKey));
            }

            // Realizar solicitud a Nominatim
            $response = Http::withHeaders([
                'User-Agent' => 'PedidosApp/1.0 (https://pedidos.test)'
            ])->get('https://nominatim.openstreetmap.org/reverse', [
                'lat' => $lat,
                'lon' => $lon,
                'zoom' => 18,
                'addressdetails' => 1,
                'format' => 'json'
            ]);

            if ($response->successful()) {
                $data = $response->json();

                // Guardar en caché por 1 hora
                Cache::put($cacheKey, $data, 3600);

                return response()->json($data);
            } else {
                Log::error('Nominatim reverse geocoding failed', [
                    'status' => $response->status(),
                    'lat' => $lat,
                    'lon' => $lon
                ]);

                return response()->json(['error' => 'Geocoding service unavailable'], 503);
            }
        } catch (\Exception $e) {
            Log::error('Nominatim reverse geocoding exception', [
                'message' => $e->getMessage(),
                'lat' => $lat,
                'lon' => $lon
            ]);

            return response()->json(['error' => 'Geocoding service error'], 500);
        }
    }

    /**
     * Método search para buscar direcciones.
     */
    public function search(Request $request)
    {
        $query = $request->query('q');

        if (!$query) {
            return response()->json(['error' => 'Query parameter is required'], 400);
        }

        try {
            $cacheKey = "search_places_" . md5($query);

            // Verificar en caché primero
            if (Cache::has($cacheKey)) {
                return response()->json(Cache::get($cacheKey));
            }

            // Realizar solicitud a Nominatim
            $response = Http::withHeaders([
                'User-Agent' => 'PedidosApp/1.0 (https://pedidos.test)'
            ])->get('https://nominatim.openstreetmap.org/search', [
                'q' => $query,
                'format' => 'json',
                'addressdetails' => 1,
                'limit' => 10
            ]);

            if ($response->successful()) {
                $data = $response->json();

                // Guardar en caché por 30 minutos
                Cache::put($cacheKey, $data, 1800);

                return response()->json($data);
            } else {
                Log::error('Nominatim search failed', [
                    'status' => $response->status(),
                    'query' => $query
                ]);

                return response()->json(['error' => 'Search service unavailable'], 503);
            }
        } catch (\Exception $e) {
            Log::error('Nominatim search exception', [
                'message' => $e->getMessage(),
                'query' => $query
            ]);

            return response()->json(['error' => 'Search service error'], 500);
        }
    }

    /**
     * Método legacy para mantener compatibilidad con llamadas antiguas.
     */
    public function nominatim(Request $request)
    {
        $query = $request->query('query');
        $lat = $request->query('lat');
        $lon = $request->query('lon');
        $type = $request->query('type', 'search');

        try {
            if ($type === 'reverse' && $lat && $lon) {
                $response = Http::get("https://nominatim.openstreetmap.org/reverse", [
                    'lat' => $lat,
                    'lon' => $lon,
                    'format' => 'json',
                    'addressdetails' => 1
                ]);
            } else {
                $response = Http::get("https://nominatim.openstreetmap.org/search", [
                    'q' => $query,
                    'format' => 'json',
                    'addressdetails' => 1,
                    'limit' => 5
                ]);
            }

            if ($response->successful()) {
                return response()->json($response->json());
            }

            return response()->json(['error' => 'Service unavailable'], 503);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Request failed'], 500);
        }
    }

    /**
     * Guardar la precisión de ubicación en la sesión.
     */
    public function saveLocationAccuracy(Request $request)
    {
        $accuracy = $request->input('accuracy');

        if (!$accuracy || !is_numeric($accuracy)) {
            return response()->json(['error' => 'Valid accuracy value is required'], 400);
        }

        // Guardar en sesión
        session(['location_accuracy' => (float) $accuracy]);

        return response()->json(['success' => true, 'accuracy' => (float) $accuracy]);
    }

    /**
     * Obtener la precisión de ubicación guardada.
     */
    public function getLocationAccuracy(Request $request)
    {
        $accuracy = session('location_accuracy', null);

        return response()->json(['accuracy' => $accuracy]);
    }

    /**
     * Método legacy para OpenCageData.
     */
    public function openCage(Request $request)
    {
        $apiKey = env('OPENCAGE_API_KEY');
        if (!$apiKey) {
            return response()->json(['error' => 'OpenCage API key not configured'], 500);
        }

        $query = $request->query('query');
        $lat = $request->query('lat');
        $lon = $request->query('lon');
        $type = $request->query('type', 'search');

        try {
            if ($type === 'reverse' && $lat && $lon) {
                $response = Http::get("https://api.opencagedata.com/geocode/v1/json", [
                    'key' => $apiKey,
                    'q' => "{$lat},{$lon}",
                    'pretty' => 1,
                    'no_annotations' => 1
                ]);
            } else {
                $response = Http::get("https://api.opencagedata.com/geocode/v1/json", [
                    'key' => $apiKey,
                    'q' => $query,
                    'pretty' => 1,
                    'no_annotations' => 1,
                    'limit' => 5
                ]);
            }

            if ($response->successful()) {
                return response()->json($response->json());
            }

            return response()->json(['error' => 'Service unavailable'], 503);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Request failed'], 500);
        }
    }
}
