<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Caja;
use App\Models\ExchangeRate;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;

class CajaExportController extends Controller
{
    public function export(Caja $caja)
    {
        $caja->load(['usuario', 'sucursal', 'pedidoPagos.pedido.user']);

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Configurar encabezado
        $sheet->setCellValue('A1', 'REPORTE DETALLADO DE CAJA');
        $sheet->mergeCells('A1:H1');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Información de la caja
        $row = 3;
        $sheet->setCellValue('A' . $row, 'Fecha:');
        $sheet->setCellValue('B' . $row, $caja->fecha->format('d/m/Y'));
        $sheet->setCellValue('D' . $row, 'Usuario:');
        $sheet->setCellValue('E' . $row, $caja->usuario->name);

        $row++;
        $sheet->setCellValue('A' . $row, 'Sucursal:');
        $sheet->setCellValue('B' . $row, $caja->sucursal->nombre);
        $sheet->setCellValue('D' . $row, 'Estado:');
        $sheet->setCellValue('E' . $row, ucfirst($caja->estado));

        // Tasa de cambio
        $tasaCambio = ExchangeRate::whereDate('created_at', $caja->fecha)->first();
        if ($tasaCambio) {
            $row++;
            $sheet->setCellValue('A' . $row, 'Tasa de Cambio:');
            $sheet->setCellValue('B' . $row, number_format($tasaCambio->usd_rate, 4) . ' Bs/$');
        }

        // Resumen financiero
        $row += 2;
        $sheet->setCellValue('A' . $row, 'RESUMEN FINANCIERO');
        $sheet->getStyle('A' . $row)->getFont()->setBold(true);

        $row++;
        $datos = [
            ['Concepto', 'Monto Principal', 'Monto Secundario'],
            ['Monto Inicial', format_money($caja->monto_inicial, false), $tasaCambio ? format_money($caja->monto_inicial * $tasaCambio->usd_rate, false) : '-'],
            ['Total Efectivo', format_money($caja->total_efectivo, false), $tasaCambio ? format_money($caja->total_efectivo * $tasaCambio->usd_rate, false) : '-'],
            ['Total Transferencias', format_money($caja->total_transferencias, false), $tasaCambio ? format_money($caja->total_transferencias * $tasaCambio->usd_rate, false) : '-'],
            ['Total Ingresos', format_money($caja->total_ingresos, false), $tasaCambio ? format_money($caja->total_ingresos * $tasaCambio->usd_rate, false) : '-'],
            ['Monto Final', format_money($caja->monto_final, false), $tasaCambio ? format_money($caja->monto_final * $tasaCambio->usd_rate, false) : '-']
        ];

        foreach ($datos as $fila) {
            $sheet->setCellValue('A' . $row, $fila[0]);
            $sheet->setCellValue('B' . $row, $fila[1]);
            $sheet->setCellValue('C' . $row, $fila[2]);
            $row++;
        }

        // Detalle de pagos
        $row += 2;
        $sheet->setCellValue('A' . $row, 'DETALLE DE PAGOS');
        $sheet->getStyle('A' . $row)->getFont()->setBold(true);

        $row++;
        $encabezados = ['Pedido', 'Cliente', 'Método', 'Monto Principal', 'Monto Secundario', 'Referencia', 'Hora'];
        foreach ($encabezados as $col => $encabezado) {
            $sheet->setCellValueByColumnAndRow($col + 1, $row, $encabezado);
        }
        $sheet->getStyle('A' . $row . ':G' . $row)->getFont()->setBold(true);

        $row++;
        foreach ($caja->pedidoPagos()->whereHas('pedido', function($query) {
            $query->where('estado', 'pagado');
        })->get() as $pago) {
            $usuario = $pago->pedido->user ?? null;
            $nombreCliente = $usuario ? $usuario->name : 'Usuario no encontrado';
            $montoUSD = format_money($pago->total_usd, false);
            $montoBolivares = $tasaCambio ? format_money($pago->total_usd * $tasaCambio->usd_rate, false) : '-';

            $sheet->setCellValue('A' . $row, '#' . $pago->pedidoId);
            $sheet->setCellValue('B' . $row, $nombreCliente);
            $sheet->setCellValue('C' . $row, ucfirst($pago->metodo_pago));
            $sheet->setCellValue('D' . $row, $montoUSD);
            $sheet->setCellValue('E' . $row, $montoBolivares);
            $sheet->setCellValue('F' . $row, $pago->referencia_bancaria_id ? 'Ref: ' . $pago->referencia_bancaria_id : '-');
            $sheet->setCellValue('G' . $row, $pago->created_at->format('H:i'));
            $row++;
        }

        // Aplicar estilos
        $sheet->getStyle('A1:G' . ($row - 1))->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
        $sheet->getColumnDimension('A')->setWidth(15);
        $sheet->getColumnDimension('B')->setWidth(25);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(12);
        $sheet->getColumnDimension('E')->setWidth(12);
        $sheet->getColumnDimension('F')->setWidth(15);
        $sheet->getColumnDimension('G')->setWidth(10);

        $writer = new Xlsx($spreadsheet);
        $filename = 'caja_' . $caja->fecha->format('Y-m-d') . '_' . $caja->numero_corte . '.xlsx';

        return response()->streamDownload(function() use ($writer) {
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        ]);
    }
}
