<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\WhatsAppService;

class WhatsAppOptimize extends Command
{
    protected $signature = 'whatsapp:optimize';
    protected $description = 'Optimizar y verificar la configuración de WhatsApp';

    public function handle()
    {
        $this->info('🚀 Optimizando configuración de WhatsApp...');
        
        // Limpiar caché de WhatsApp
        $this->info('🧹 Limpiando caché...');
        cache()->forget('whatsapp_status_main');
        cache()->forget('whatsapp_connection_test');
        cache()->forget('whatsapp_status_dashboard');
        
        // Verificar conectividad
        $this->info('🔍 Verificando conectividad...');
        $service = new WhatsAppService();
        
        $start = microtime(true);
        $result = $service->testConnection();
        $duration = round((microtime(true) - $start) * 1000, 2);
        
        if ($result['success']) {
            $this->info("✅ API conectada en {$duration}ms");
        } else {
            $this->error("❌ Error de conexión: {$result['error']} ({$duration}ms)");
        }
        
        // Verificar estado
        $this->info('📊 Verificando estado...');
        $start = microtime(true);
        $status = $service->getStatus();
        $duration = round((microtime(true) - $start) * 1000, 2);
        
        if ($status['success']) {
            $this->info("✅ Estado obtenido en {$duration}ms");
            $this->line("   Estado: {$status['status']}");
            $this->line("   Listo: " . ($status['isReady'] ? 'Sí' : 'No'));
        } else {
            $this->error("❌ Error obteniendo estado: {$status['error']} ({$duration}ms)");
        }
        
        // Mostrar configuración actual
        $this->info('⚙️ Configuración actual:');
        $this->table(['Configuración', 'Valor'], [
            ['API URL', config('whatsapp.api_url')],
            ['Timeout Conexión', config('whatsapp.timeouts.connection') . 's'],
            ['Timeout Estado', config('whatsapp.timeouts.status') . 's'],
            ['Timeout Mensaje', config('whatsapp.timeouts.send_message') . 's'],
            ['TTL Caché', config('whatsapp.cache.status_ttl') . 's'],
        ]);
        
        $this->info('✨ Optimización completada');
        
        return 0;
    }
}