<?php

namespace App\Console\Commands;

use App\Services\WhatsAppHealthService;
use Illuminate\Console\Command;

class WhatsAppHealthCheck extends Command
{
    protected $signature = 'whatsapp:health {--json : Output as JSON}';
    protected $description = 'Verificar el estado de salud de la API de WhatsApp';

    public function handle()
    {
        $healthService = new WhatsAppHealthService();
        
        $this->info('🔍 Verificando salud de WhatsApp API...');
        $this->newLine();
        
        $health = $healthService->healthCheck();
        
        if ($this->option('json')) {
            $this->line(json_encode($health, JSON_PRETTY_PRINT));
            return;
        }
        
        // Mostrar estado general
        $statusEmoji = match($health['overall_status']) {
            'excellent' => '🟢',
            'good' => '🟡',
            'fair' => '🟠',
            default => '🔴'
        };
        
        $this->info("Estado General: {$statusEmoji} {$health['overall_status']} ({$health['score']}%)");
        $this->newLine();
        
        // Mostrar detalles de cada verificación
        foreach ($health['checks'] as $name => $check) {
            $emoji = match($check['status']) {
                'healthy' => '✅',
                'warning' => '⚠️',
                default => '❌'
            };
            
            $this->line("{$emoji} " . ucfirst(str_replace('_', ' ', $name)) . ": {$check['message']}");
        }
        
        $this->newLine();
        
        // Mostrar métricas de rendimiento
        try {
            $metrics = $healthService->getPerformanceMetrics();
            $this->info('📊 Métricas de Rendimiento:');
            $this->line("   Tiempo promedio de respuesta: {$metrics['avg_response_time']}ms");
            $this->line("   Tasa de éxito: {$metrics['success_rate']}%");
            $this->line("   Mensajes enviados hoy: {$metrics['daily_message_count']}");
            $this->line("   Tasa de error: {$metrics['error_rate']}%");
        } catch (\Exception $e) {
            $this->warn('No se pudieron obtener métricas de rendimiento');
        }
        
        return $health['overall_status'] === 'excellent' ? 0 : 1;
    }
}
